package org.dromara.zjk.expert.expertRecommend.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.expert.expertRecommend.domain.bo.ZjkExpertRecommendBo;
import org.dromara.zjk.expert.expertRecommend.domain.vo.ZjkExpertRecommendVo;
import org.dromara.zjk.expert.expertRecommend.service.IZjkExpertRecommendService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.dromara.zjk.zwy.annotation.EncryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 专家推荐信息
 * 前端访问路由地址为:/expert/expertRecommend
 *
 * @author zhangzhou
 * @date 2024-11-10
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/expertRecommend")
public class ZjkExpertRecommendController extends BaseController {

    private final IZjkExpertRecommendService zjkExpertRecommendService;

    /**
     * 查询专家推荐信息列表
     */
    @DecryptOperation
    @SaCheckPermission("expert:expertRecommend:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkExpertRecommendVo> list(ZjkExpertRecommendBo bo, PageQuery pageQuery) {
        return zjkExpertRecommendService.queryPageList(bo, pageQuery);
    }

    /**
     * 查询专家推荐信息列表
     */
    @DecryptOperation
    @SaCheckPermission("expert:expertRecommend:list")
    @GetMapping("/queryExpertRecommendList")
    public TableDataInfo<ZjkExpertRecommendVo> queryExpertRecommendList(ZjkExpertRecommendBo bo, PageQuery pageQuery) {
        return zjkExpertRecommendService.queryExpertRecommendList(bo, pageQuery);
    }

    /**
     * 导出专家推荐信息列表
     */
    @DecryptOperation
    @SaCheckPermission("expert:expertRecommend:export")
    @Log(title = "专家推荐信息", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkExpertRecommendBo bo, HttpServletResponse response) {
        List<ZjkExpertRecommendVo> list = zjkExpertRecommendService.queryList(bo);
        ExcelUtil.exportExcel(list, "专家推荐信息", ZjkExpertRecommendVo.class, response);
    }

    /**
     * 获取专家推荐信息详细信息
     *
     * @param id 主键
     */
    @DecryptOperation
    @SaCheckPermission("expert:expertRecommend:query")
    @GetMapping("/{id}")
    public R<ZjkExpertRecommendVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkExpertRecommendService.queryById(id));
    }

    /**
     * 新增专家推荐信息
     */
    @EncryptOperation
    @SaCheckPermission("expert:expertRecommend:add")
    @Log(title = "专家推荐信息", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkExpertRecommendBo bo) {
        return toAjax(zjkExpertRecommendService.insertByBo(bo));
    }

    /**
     * 修改专家推荐信息
     */
    @EncryptOperation
    @SaCheckPermission("expert:expertRecommend:edit")
    @Log(title = "专家推荐信息", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkExpertRecommendBo bo) {
        return toAjax(zjkExpertRecommendService.updateByBo(bo));
    }

    /**
     * 删除专家推荐信息
     *
     * @param ids 主键串
     */
    @SaCheckPermission("expert:expertRecommend:remove")
    @Log(title = "专家推荐信息", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkExpertRecommendService.deleteWithValidByIds(List.of(ids), true));
    }
}
