package org.dromara.zjk.learning_world.controller;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.learning_world.domain.bo.ZjkLearningActivityBo;
import org.dromara.zjk.learning_world.domain.excel.ZjkLearningActivityExcel;
import org.dromara.zjk.learning_world.domain.vo.ZjkLearningActivityVo;
import org.dromara.zjk.learning_world.service.IZjkLearningActivityService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 活动管理
 * 前端访问路由地址为:/zjk/learningActivity
 *
 * @author ygh
 * @date 2024-11-08
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningActivity")
public class ZjkLearningActivityController extends BaseController {

    private final IZjkLearningActivityService zjkLearningActivityService;

    /**
     * 查询活动管理列表
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkLearningActivityVo> list(ZjkLearningActivityBo bo, PageQuery pageQuery) {
        return zjkLearningActivityService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出活动管理列表
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "活动管理", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkLearningActivityBo bo, HttpServletResponse response) {
        List<ZjkLearningActivityExcel> list = zjkLearningActivityService.exportList(bo);
        ExcelUtil.exportExcel(list, "活动管理", ZjkLearningActivityExcel.class, response);
    }

    /**
     * 获取活动管理详细信息
     *
     * @param id 主键
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkLearningActivityVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkLearningActivityService.queryById(id));
    }

    /**
     * 新增活动管理
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "活动管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkLearningActivityBo bo) {
        return toAjax(zjkLearningActivityService.insertByBo(bo));
    }

    /**
     * 修改活动管理
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "活动管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkLearningActivityBo bo) {
        return toAjax(zjkLearningActivityService.updateByBo(bo));
    }

    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "活动管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update/status")
    public R<Void> editStatus(@Validated(EditGroup.class) @RequestBody ZjkLearningActivityBo bo) {
        return toAjax(zjkLearningActivityService.updateStatus(bo));
    }

    /**
     * 删除活动管理
     *
     * @param ids 主键串
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "活动管理", businessType = BusinessType.DELETE)
    @GetMapping("/deleted/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkLearningActivityService.deleteWithValidByIds(List.of(ids), true));
    }
}
