package org.dromara.zjk.learning_world.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.zjk.learning_world.domain.excel.ZjkLearningResourceExcel;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.learning_world.domain.vo.ZjkLearningResourceVo;
import org.dromara.zjk.learning_world.domain.bo.ZjkLearningResourceBo;
import org.dromara.zjk.learning_world.service.IZjkLearningResourceService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 资源管理
 * 前端访问路由地址为:/zjk/learningResource
 *
 * @author ygh
 * @date 2024-11-08
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningResource")
public class ZjkLearningResourceController extends BaseController {

    private final IZjkLearningResourceService zjkLearningResourceService;

    /**
     * 查询资源管理列表
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkLearningResourceVo> list(ZjkLearningResourceBo bo, PageQuery pageQuery) {
        return zjkLearningResourceService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出资源管理列表
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "资源管理", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkLearningResourceBo bo, HttpServletResponse response) {
        List<ZjkLearningResourceExcel> list = zjkLearningResourceService.exportList(bo);
        ExcelUtil.exportExcel(list, "资源管理", ZjkLearningResourceExcel.class, response);
    }

    /**
     * 获取资源管理详细信息
     *
     * @param id 主键
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkLearningResourceVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkLearningResourceService.queryById(id));
    }

    /**
     * 新增资源管理
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "资源管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkLearningResourceBo bo) {
        return toAjax(zjkLearningResourceService.insertByBo(bo));
    }

    /**
     * 修改资源管理
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "资源管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkLearningResourceBo bo) {
        return toAjax(zjkLearningResourceService.updateByBo(bo));
    }


    /**
     * 修改资源管理
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "资源管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update/status")
    public R<Void> updateStatus(@Validated(EditGroup.class) @RequestBody ZjkLearningResourceBo bo) {
        return toAjax(zjkLearningResourceService.updateStatus(bo));
    }

    /**
     * 删除资源管理
     *
     * @param ids 主键串
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "资源管理", businessType = BusinessType.DELETE)
    @GetMapping("/deleted/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkLearningResourceService.deleteWithValidByIds(List.of(ids), true));
    }
}
