package org.dromara.zjk.learning_world.controller;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.learning_world.domain.bo.ZjkLearningToolBo;
import org.dromara.zjk.learning_world.domain.vo.ZjkLearningToolVo;
import org.dromara.zjk.learning_world.service.IZjkLearningToolService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 学习工具管理
 * 前端访问路由地址为:/zjk/learningTool
 *
 * @author ygh
 * @date 2024-11-10
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningTool")
public class ZjkLearningToolController extends BaseController {

    private final IZjkLearningToolService zjkLearningToolService;

    /**
     * 查询学习工具管理列表
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkLearningToolVo>  list(ZjkLearningToolBo bo, PageQuery pageQuery) {
        return zjkLearningToolService.queryPageList(bo, pageQuery);
    }


    /**
     * 新增学习工具管理
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "学习工具管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkLearningToolBo bo) {
        return toAjax(zjkLearningToolService.insertByBo(bo));
    }

    /**
     * 修改学习工具管理
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "学习工具管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkLearningToolBo bo) {
        return toAjax(zjkLearningToolService.updateByBo(bo));
    }

    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "学习工具管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update-status")
    public R<Void> editStatus(@Validated(EditGroup.class) @RequestBody ZjkLearningToolBo bo) {
        return toAjax(zjkLearningToolService.editStatus(bo));
    }

    /**
     * 学习工具管理
     *
     * @param id 主键串
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkLearningToolVo> getById(@NotNull(message = "主键不能为空")
                                            @PathVariable Long id) {
        return R.ok(zjkLearningToolService.queryById(id));
    }

    /**
     * 删除学习工具管理
     *
     * @param ids 主键串
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "学习工具管理", businessType = BusinessType.DELETE)
    @GetMapping("/deleted/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkLearningToolService.deleteWithValidByIds(List.of(ids), true));
    }
}
