package org.dromara.zjk.service.impl;

import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.mybatis.core.page.PageQuery;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.dromara.zjk.domain.bo.ZjkExpertHistoryBo;
import org.dromara.zjk.domain.vo.ZjkExpertHistoryVo;
import org.dromara.zjk.domain.ZjkExpertHistory;
import org.dromara.zjk.mapper.ZjkExpertHistoryMapper;
import org.dromara.zjk.service.IZjkExpertHistoryService;

import java.util.List;
import java.util.Map;
import java.util.Collection;

/**
 * 专家记录变更历史表Service业务层处理
 *
 * @author zhangzhou
 * @date 2024-11-02
 */
@RequiredArgsConstructor
@Service
public class ZjkExpertHistoryServiceImpl implements IZjkExpertHistoryService {

    private final ZjkExpertHistoryMapper baseMapper;

    /**
     * 查询专家记录变更历史表
     *
     * @param id 主键
     * @return 专家记录变更历史表
     */
    @Override
    public ZjkExpertHistoryVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询专家记录变更历史表列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 专家记录变更历史表分页列表
     */
    @Override
    public TableDataInfo<ZjkExpertHistoryVo> queryPageList(ZjkExpertHistoryBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkExpertHistory> lqw = buildQueryWrapper(bo);
        Page<ZjkExpertHistoryVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的专家记录变更历史表列表
     *
     * @param bo 查询条件
     * @return 专家记录变更历史表列表
     */
    @Override
    public List<ZjkExpertHistoryVo> queryList(ZjkExpertHistoryBo bo) {
        LambdaQueryWrapper<ZjkExpertHistory> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkExpertHistory> buildQueryWrapper(ZjkExpertHistoryBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkExpertHistory> lqw = Wrappers.lambdaQuery();
        lqw.eq(StringUtils.isNotBlank(bo.getOldData()), ZjkExpertHistory::getOldData, bo.getOldData());
        lqw.eq(StringUtils.isNotBlank(bo.getNewData()), ZjkExpertHistory::getNewData, bo.getNewData());
        lqw.eq(StringUtils.isNotBlank(bo.getModifiedFields()), ZjkExpertHistory::getModifiedFields, bo.getModifiedFields());
        lqw.eq(bo.getDeleteFlag() != null, ZjkExpertHistory::getDeleteFlag, bo.getDeleteFlag());
        return lqw;
    }

    /**
     * 新增专家记录变更历史表
     *
     * @param bo 专家记录变更历史表
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkExpertHistoryBo bo) {
        ZjkExpertHistory add = MapstructUtils.convert(bo, ZjkExpertHistory.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改专家记录变更历史表
     *
     * @param bo 专家记录变更历史表
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkExpertHistoryBo bo) {
        ZjkExpertHistory update = MapstructUtils.convert(bo, ZjkExpertHistory.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkExpertHistory entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除专家记录变更历史表信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }
}
