package org.dromara.zjk.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.seata.common.util.CollectionUtils;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.exception.ServiceException;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.zjk.domain.ZjkExpertMoneyConfig;
import org.dromara.zjk.domain.bo.ZjkExpertMoneyConfigBo;
import org.dromara.zjk.domain.vo.SelectOptionVo;
import org.dromara.zjk.domain.vo.ZjkExpertMoneyConfigVo;
import org.dromara.zjk.mapper.ZjkExpertMoneyConfigMapper;
import org.dromara.zjk.service.IZjkExpertMoneyConfigService;
import org.springframework.stereotype.Service;

import java.rmi.server.ServerCloneException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 专家费用管理Service业务层处理
 *
 * @author by
 * @date 2024-11-25
 */
@RequiredArgsConstructor
@Service
public class ZjkExpertMoneyConfigServiceImpl implements IZjkExpertMoneyConfigService {

    private final ZjkExpertMoneyConfigMapper baseMapper;

    /**
     * 查询专家费用管理
     *
     * @param moneyId 主键
     * @return 专家费用管理
     */
    @Override
    public ZjkExpertMoneyConfigVo queryById(Long moneyId){
        return baseMapper.selectVoById(moneyId);
    }

    /**
     * 分页查询专家费用管理列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 专家费用管理分页列表
     */
    @Override
    public TableDataInfo<ZjkExpertMoneyConfigVo> queryPageList(ZjkExpertMoneyConfigBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkExpertMoneyConfig> lqw = buildQueryWrapper(bo);
        Page<ZjkExpertMoneyConfigVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的专家费用管理列表
     *
     * @param bo 查询条件
     * @return 专家费用管理列表
     */
    @Override
    public List<ZjkExpertMoneyConfigVo> queryList(ZjkExpertMoneyConfigBo bo) {
        LambdaQueryWrapper<ZjkExpertMoneyConfig> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkExpertMoneyConfig> buildQueryWrapper(ZjkExpertMoneyConfigBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkExpertMoneyConfig> lqw = Wrappers.lambdaQuery();
        lqw.like(StringUtils.isNotBlank(bo.getMoneyType()), ZjkExpertMoneyConfig::getMoneyType, bo.getMoneyType());
        lqw.like(StringUtils.isNotBlank(bo.getMoneyCycle()), ZjkExpertMoneyConfig::getMoneyCycle, bo.getMoneyCycle());
        lqw.like(StringUtils.isNotBlank(bo.getMoneyMethod()), ZjkExpertMoneyConfig::getMoneyMethod, bo.getMoneyMethod());
        lqw.eq(bo.getMoneyNumber() != null, ZjkExpertMoneyConfig::getMoneyNumber, bo.getMoneyNumber());
        lqw.in(CollectionUtils.isNotEmpty(bo.getMoneyTypeSet()), ZjkExpertMoneyConfig::getMoneyType, bo.getMoneyTypeSet());
        return lqw;
    }

    /**
     * 新增专家费用管理
     *
     * @param bo 专家费用管理
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkExpertMoneyConfigBo bo) {

        ZjkExpertMoneyConfig add = MapstructUtils.convert(bo, ZjkExpertMoneyConfig.class);
        validEntityBeforeSave(add);
        String moneyType = add.getMoneyType();

        LambdaQueryWrapper<ZjkExpertMoneyConfig> wrapper = Wrappers.lambdaQuery();
        wrapper.eq(ZjkExpertMoneyConfig::getMoneyType, moneyType);

        List<ZjkExpertMoneyConfig> zjkExpertMoneyConfigs = baseMapper.selectList(wrapper);
        if(CollectionUtil.isNotEmpty(zjkExpertMoneyConfigs)){
            throw new ServiceException("类型已存在");
        }
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setMoneyId(add.getMoneyId());
        }
        return flag;
    }

    /**
     * 修改专家费用管理
     *
     * @param bo 专家费用管理
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkExpertMoneyConfigBo bo) {
        ZjkExpertMoneyConfig update = MapstructUtils.convert(bo, ZjkExpertMoneyConfig.class);
        LambdaQueryWrapper<ZjkExpertMoneyConfig> wrapper = Wrappers.lambdaQuery();
        wrapper.eq(ZjkExpertMoneyConfig::getMoneyType, bo.getMoneyType());
        wrapper.ne(ZjkExpertMoneyConfig::getMoneyId, bo.getMoneyId());
        List<ZjkExpertMoneyConfig> zjkExpertMoneyConfigs = baseMapper.selectList(wrapper);
        if(CollectionUtil.isNotEmpty(zjkExpertMoneyConfigs)){
            throw new ServiceException("类型已存在");
        }
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkExpertMoneyConfig entity) {

        //TODO 做一些数据校验,如唯一约束

    }

    /**
     * 校验并批量删除专家费用管理信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public List<SelectOptionVo> getExpertMoneyConfig() {
        return baseMapper.getExpertMoneyConfig();
    }

    @Override
    public Boolean imports(List<ZjkExpertMoneyConfigVo> vos) {
        Map<String, List<ZjkExpertMoneyConfigVo>> typeNameList = vos.stream().collect(Collectors.groupingBy(ZjkExpertMoneyConfigVo::getMoneyType));
        Set<String> typeNameSet = typeNameList.keySet();
        for (String typeName : typeNameList.keySet()) {
            List<ZjkExpertMoneyConfigVo> zjkExpertMoneyConfigVos = typeNameList.get(typeName);
            if(zjkExpertMoneyConfigVos.size()>1){
                throw new ServiceException(typeName+"已重复，导入文件中重复");
            }
        }
        if(CollectionUtil.isEmpty(typeNameSet)){
            throw new ServiceException("导入文件为空");
        }
        LambdaQueryWrapper<ZjkExpertMoneyConfig> lqw = Wrappers.lambdaQuery();
        lqw.in(ZjkExpertMoneyConfig::getMoneyType, typeNameSet);
        List<ZjkExpertMoneyConfig> zjkExpertMoneyConfigs = baseMapper.selectList(lqw);
        if(CollectionUtil.isNotEmpty(zjkExpertMoneyConfigs)){
            String result = zjkExpertMoneyConfigs.stream()
                .map(ZjkExpertMoneyConfig::getMoneyType)
                .collect(Collectors.joining(", "));

            throw new ServiceException(result+"已重复，导入文件中重复");
        }
        List<ZjkExpertMoneyConfig> entityList = BeanUtil.copyToList(vos, ZjkExpertMoneyConfig.class);
       return  baseMapper.insertBatch(entityList);
    }
}
