package org.dromara.zjk.service.impl;

import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.mybatis.core.page.PageQuery;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.dromara.zjk.domain.bo.ZjkExpertProfessionalBo;
import org.dromara.zjk.domain.vo.ZjkExpertProfessionalVo;
import org.dromara.zjk.domain.ZjkExpertProfessional;
import org.dromara.zjk.mapper.ZjkExpertProfessionalMapper;
import org.dromara.zjk.service.IZjkExpertProfessionalService;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Collection;

/**
 * 专家库技术资格关系表Service业务层处理
 *
 * @author zhangzhou
 * @date 2024-11-01
 */
@RequiredArgsConstructor
@Service
public class ZjkExpertProfessionalServiceImpl implements IZjkExpertProfessionalService {

    private final ZjkExpertProfessionalMapper baseMapper;

    /**
     * 查询专家库技术资格关系表
     *
     * @param id 主键
     * @return 专家库技术资格关系表
     */
    @Override
    public ZjkExpertProfessionalVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询专家库技术资格关系表列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 专家库技术资格关系表分页列表
     */
    @Override
    public TableDataInfo<ZjkExpertProfessionalVo> queryPageList(ZjkExpertProfessionalBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkExpertProfessional> lqw = buildQueryWrapper(bo);
        Page<ZjkExpertProfessionalVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的专家库技术资格关系表列表
     *
     * @param bo 查询条件
     * @return 专家库技术资格关系表列表
     */
    @Override
    public List<ZjkExpertProfessionalVo> queryList(ZjkExpertProfessionalBo bo) {
        LambdaQueryWrapper<ZjkExpertProfessional> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkExpertProfessional> buildQueryWrapper(ZjkExpertProfessionalBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkExpertProfessional> lqw = Wrappers.lambdaQuery();
        lqw.eq(bo.getUserId() != null, ZjkExpertProfessional::getUserId, bo.getUserId());
        lqw.eq(bo.getExpertId() != null, ZjkExpertProfessional::getExpertId, bo.getExpertId());
        lqw.eq(StringUtils.isNotBlank(bo.getProfessional()), ZjkExpertProfessional::getProfessional, bo.getProfessional());
        lqw.eq(bo.getProfessionalDate() != null, ZjkExpertProfessional::getProfessionalDate, bo.getProfessionalDate());
        lqw.eq(bo.getStatus() != null, ZjkExpertProfessional::getStatus, bo.getStatus());
        return lqw;
    }

    /**
     * 新增专家库技术资格关系表
     *
     * @param bo 专家库技术资格关系表
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkExpertProfessionalBo bo) {
        ZjkExpertProfessional add = MapstructUtils.convert(bo, ZjkExpertProfessional.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改专家库技术资格关系表
     *
     * @param bo 专家库技术资格关系表
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkExpertProfessionalBo bo) {
        ZjkExpertProfessional update = MapstructUtils.convert(bo, ZjkExpertProfessional.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkExpertProfessional entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除专家库技术资格关系表信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public List<String> professionalList() {
        return baseMapper.professionalList();
    }


    /**
     * 根据用户 id  专家 id 进行 新增专业技术资格信息
     * @param zjkExpertProfessionals
     * @param expertId
     * @param userId
     */
    @Override
    public void handleExpertProfessionals(List<ZjkExpertProfessional> zjkExpertProfessionals, Long expertId, Long userId) {
        // 删除旧的专业技术资格信息
        baseMapper.delete(new LambdaQueryWrapper<ZjkExpertProfessional>()
            .eq(ZjkExpertProfessional::getExpertId, expertId)
            .eq(ZjkExpertProfessional::getUserId, userId)
        );
        if (zjkExpertProfessionals!= null&&!zjkExpertProfessionals.isEmpty()){
            // 设置 userId 和 expertId 并批量插入或更新新数据
            zjkExpertProfessionals.forEach(professional -> {
                professional.setUserId(userId);
                professional.setExpertId(expertId);
            });
            baseMapper.insertOrUpdateBatch(zjkExpertProfessionals);
        }
    }

    /**
     * 根据用户 id 专家 id 获取专家库技术资格关系表对象
     * @param userId
     * @param expertId
     * @return
     */
    @Override
    public List<ZjkExpertProfessional> populateExpertProfessionalList(Long userId, Long expertId) {
        List<ZjkExpertProfessional> zjkExpertProfessionals = baseMapper.selectList(
            new LambdaQueryWrapper<ZjkExpertProfessional>()
                .eq(ZjkExpertProfessional::getExpertId, expertId)
                .eq(ZjkExpertProfessional::getUserId, userId)
        );
        return zjkExpertProfessionals.isEmpty() ? new ArrayList<>() : zjkExpertProfessionals;
    }
}
