package org.dromara.zjk.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.zjk.domain.ZjkLearningMaterials;
import org.dromara.zjk.domain.bo.ZjkLearningMaterialsBo;
import org.dromara.zjk.domain.vo.ZjkLearningMaterialsVo;
import org.dromara.zjk.mapper.ZjkLearningMaterialsMapper;
import org.dromara.zjk.service.IZjkLearningMaterialsService;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 信用资料Service业务层处理
 *
 * @author ygh
 * @date 2024-11-26
 */
@RequiredArgsConstructor
@Service
public class ZjkLearningMaterialsServiceImpl implements IZjkLearningMaterialsService {

    private final ZjkLearningMaterialsMapper baseMapper;

    /**
     * 查询信用资料
     *
     * @param id 主键
     * @return 信用资料
     */
    @Override
    public ZjkLearningMaterialsVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询信用资料列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 信用资料分页列表
     */
    @Override
    public TableDataInfo<ZjkLearningMaterialsVo> queryPageList(ZjkLearningMaterialsBo bo, PageQuery pageQuery) {
        Page<ZjkLearningMaterialsVo> result = baseMapper.selectByAll(pageQuery.build(), bo);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的信用资料列表
     *
     * @param bo 查询条件
     * @return 信用资料列表
     */
    @Override
    public List<ZjkLearningMaterialsVo> queryList(ZjkLearningMaterialsBo bo) {
        LambdaQueryWrapper<ZjkLearningMaterials> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkLearningMaterials> buildQueryWrapper(ZjkLearningMaterialsBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkLearningMaterials> lqw = Wrappers.lambdaQuery();
        lqw.eq(StringUtils.isNotBlank(bo.getMaterialsId()), ZjkLearningMaterials::getMaterialsId, bo.getMaterialsId());
        lqw.like(StringUtils.isNotBlank(bo.getMaterialsName()), ZjkLearningMaterials::getMaterialsName, bo.getMaterialsName());
        lqw.le(bo.getStartTime() != null, ZjkLearningMaterials::getCreateTime, bo.getStartTime());
        lqw.ge(bo.getEndTime() != null, ZjkLearningMaterials::getCreateTime, bo.getEndTime());
        lqw.eq(StringUtils.isNotBlank(bo.getMaterialsCount()), ZjkLearningMaterials::getMaterialsCount, bo.getMaterialsCount());
        lqw.eq(StringUtils.isNotBlank(bo.getStatus()), ZjkLearningMaterials::getStatus, bo.getStatus());
        lqw.orderByDesc(ZjkLearningMaterials::getCreateTime);
        return lqw;
    }

    /**
     * 新增信用资料
     *
     * @param bo 信用资料
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkLearningMaterialsBo bo) {
        ZjkLearningMaterials add = MapstructUtils.convert(bo, ZjkLearningMaterials.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改信用资料
     *
     * @param bo 信用资料
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkLearningMaterialsBo bo) {
        ZjkLearningMaterials update = MapstructUtils.convert(bo, ZjkLearningMaterials.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkLearningMaterials entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除信用资料信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }
}
