package org.dromara.zjk.statement.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.statement.domain.vo.ZjkMoneyCollectStatementVo;
import org.dromara.zjk.statement.domain.bo.ZjkMoneyCollectStatementBo;
import org.dromara.zjk.statement.service.IZjkMoneyCollectStatementService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 报表统计
 * 前端访问路由地址为:/zjk/moneyCollectStatement
 *
 * @author wt
 * @date 2024-12-16
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/moneyCollectStatement")
public class ZjkMoneyCollectStatementController extends BaseController {

    private final IZjkMoneyCollectStatementService zjkMoneyCollectStatementService;

    /**
     * 查询报表统计列表
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkMoneyCollectStatementVo> list(ZjkMoneyCollectStatementBo bo, PageQuery pageQuery) {
        return zjkMoneyCollectStatementService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出报表统计列表
     */
    @SaCheckPermission("zjk:moneyCollectStatement:export")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "报表统计", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkMoneyCollectStatementBo bo, HttpServletResponse response) {
        List<ZjkMoneyCollectStatementVo> list = zjkMoneyCollectStatementService.queryList(bo);
        ExcelUtil.exportExcel(list, "报表统计", ZjkMoneyCollectStatementVo.class, response);
    }

    /**
     * 获取报表统计详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("zjk:moneyCollectStatement:query")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkMoneyCollectStatementVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkMoneyCollectStatementService.queryById(id));
    }

    /**
     * 新增报表统计
     */
    @SaCheckPermission("zjk:moneyCollectStatement:add")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "报表统计", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkMoneyCollectStatementBo bo) {
        return toAjax(zjkMoneyCollectStatementService.insertByBo(bo));
    }

    /**
     * 修改报表统计
     */
    @SaCheckPermission("zjk:moneyCollectStatement:edit")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "报表统计", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkMoneyCollectStatementBo bo) {
        return toAjax(zjkMoneyCollectStatementService.updateByBo(bo));
    }

    /**
     * 删除报表统计
     *
     * @param ids 主键串
     */
    @SaCheckPermission("zjk:moneyCollectStatement:remove")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "报表统计", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkMoneyCollectStatementService.deleteWithValidByIds(List.of(ids), true));
    }
}
