package org.dromara.zjk.learning_world.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.zjk.learning_world.domain.bo.ZjkProblemBo;
import org.dromara.zjk.learning_world.domain.vo.ZjkProblemVo;
import org.dromara.zjk.learning_world.service.IZjkProblemService;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 问题管理
 * 前端访问路由地址为:/zjk/problem
 *
 * @author wt
 * @date 2024-11-20
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/problem")
public class ZjkProblemController extends BaseController {

    private final IZjkProblemService zjkProblemService;

    /**
     * 查询问题管理列表
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkProblemVo> list(ZjkProblemBo bo, PageQuery pageQuery) {
        return zjkProblemService.queryPageList(bo, pageQuery);
    }

    /**
     * 获取问题管理详细信息
     *
     * @param id 主键
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkProblemVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkProblemService.queryById(id));
    }

    /**
     * 新增问题管理
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "问题管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkProblemBo bo) {
        return toAjax(zjkProblemService.insertByBo(bo));
    }
    /**
     * 回复问题管理
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "问题管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/reply")
    public R<Void> reply(@Validated(EditGroup.class) @RequestBody ZjkProblemBo bo) {
        return toAjax(zjkProblemService.reply(bo));
    }

    /**
     * 修改问题管理
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "问题管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkProblemBo bo) {
        return toAjax(zjkProblemService.updateByBo(bo));
    }

    /**
     * 删除问题管理
     *
     * @param ids 主键串
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "问题管理", businessType = BusinessType.DELETE)
    @PostMapping("/delete")
    public R<Void> remove(@RequestBody ZjkProblemBo bo) {
        Long[] ids = bo.getIds();
        return toAjax(zjkProblemService.deleteWithValidByIds(List.of(ids), true));
    }
}
