package org.dromara.zjk.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.exception.ServiceException;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.satoken.utils.LoginHelper;
import org.dromara.zjk.domain.ZjkProductSummarizeLog;
import org.dromara.zjk.domain.ZjkProjectEvaluation;
import org.dromara.zjk.domain.bo.ZjkProductSummarizeLogBo;
import org.dromara.zjk.domain.vo.AuditVo;
import org.dromara.zjk.domain.vo.ZjkProductSummarizeLogVo;
import org.dromara.zjk.enums.ApprovalProcessBusinessTypeEnum;
import org.dromara.zjk.enums.ApprovalProcessTypeEnum;
import org.dromara.zjk.enums.SummarizeStatusEnum;
import org.dromara.zjk.mapper.ZjkProductSummarizeLogMapper;
import org.dromara.zjk.mapper.ZjkProjectEvaluationMapper;
import org.dromara.zjk.service.IZjkApprovalProcessLogsService;
import org.dromara.zjk.service.IZjkProductSummarizeLogService;
import org.dromara.zjk.service.IZjkProductSummarizeService;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 总结日志表Service业务层处理
 *
 * @author ygh
 * @date 2024-11-04
 */
@RequiredArgsConstructor
@Service
public class ZjkProductSummarizeLogServiceImpl implements IZjkProductSummarizeLogService {

    private final ZjkProductSummarizeLogMapper baseMapper;

    private final IZjkProductSummarizeService summarizeService;

    private final IZjkApprovalProcessLogsService zjkApprovalProcessLogsService;

    private final ZjkProjectEvaluationMapper zjkProjectEvaluationMapper;


    /**
     * 查询总结日志表
     *
     * @param id 主键
     * @return 总结日志表
     */
    @Override
    public ZjkProductSummarizeLogVo queryById(Long id) {
        ZjkProductSummarizeLogVo byId = baseMapper.getById(id);
        LambdaQueryWrapper<ZjkProjectEvaluation> lambdaQueryWrapper = new LambdaQueryWrapper<ZjkProjectEvaluation>();
        lambdaQueryWrapper.eq(ZjkProjectEvaluation::getProductSummarizeId,byId.getId());
        ZjkProjectEvaluation zjkProjectEvaluation = zjkProjectEvaluationMapper.selectOne(lambdaQueryWrapper);
        if(zjkProjectEvaluation!=null) {
            byId.setZjkProjectEvaluation(zjkProjectEvaluation);
        }
        return byId;
    }

    /**
     * 分页查询总结日志表列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 总结日志表分页列表
     */
    @Override
    public TableDataInfo<ZjkProductSummarizeLogVo> queryPageList(ZjkProductSummarizeLogBo bo, PageQuery pageQuery) {
        List<ZjkProductSummarizeLogVo> zjkProductSummarizeLogVos = baseMapper.queryPage(pageQuery.build(), bo);
        return TableDataInfo.build(zjkProductSummarizeLogVos);
    }

    /**
     * 查询符合条件的总结日志表列表
     *
     * @param bo 查询条件
     * @return 总结日志表列表
     */
    @Override
    public List<ZjkProductSummarizeLogVo> queryList(ZjkProductSummarizeLogBo bo) {
        LambdaQueryWrapper<ZjkProductSummarizeLog> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkProductSummarizeLog> buildQueryWrapper(ZjkProductSummarizeLogBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkProductSummarizeLog> lqw = Wrappers.lambdaQuery();
        lqw.eq(bo.getSummarizeId() != null, ZjkProductSummarizeLog::getSummarizeId, bo.getSummarizeId());
        lqw.eq(bo.getReviewId() != null, ZjkProductSummarizeLog::getReviewId, bo.getReviewId());
        lqw.like(StringUtils.isNotBlank(bo.getProductSummarizeName()), ZjkProductSummarizeLog::getProductSummarizeName, bo.getProductSummarizeName());
        lqw.eq(StringUtils.isNotBlank(bo.getProductSummarizeType()), ZjkProductSummarizeLog::getProductSummarizeType, bo.getProductSummarizeType());
        lqw.eq(StringUtils.isNotBlank(bo.getProductGeneral()), ZjkProductSummarizeLog::getProductGeneral, bo.getProductGeneral());
        lqw.eq(StringUtils.isNotBlank(bo.getReviewScope()), ZjkProductSummarizeLog::getReviewScope, bo.getReviewScope());
        lqw.eq(StringUtils.isNotBlank(bo.getReviewConclusion()), ZjkProductSummarizeLog::getReviewConclusion, bo.getReviewConclusion());
        lqw.eq(StringUtils.isNotBlank(bo.getFeedback()), ZjkProductSummarizeLog::getFeedback, bo.getFeedback());
        lqw.eq(StringUtils.isNotBlank(bo.getSubsequent()), ZjkProductSummarizeLog::getSubsequent, bo.getSubsequent());
        lqw.eq(StringUtils.isNotBlank(bo.getAttachment()), ZjkProductSummarizeLog::getAttachment, bo.getAttachment());
        lqw.eq(StringUtils.isNotBlank(bo.getExpertIds()), ZjkProductSummarizeLog::getExpertIds, bo.getExpertIds());
        lqw.eq(StringUtils.isNotBlank(bo.getStatus()), ZjkProductSummarizeLog::getStatus, bo.getStatus());
        lqw.eq(StringUtils.isNotBlank(bo.getAuditStatus()), ZjkProductSummarizeLog::getAuditStatus, bo.getAuditStatus());
        lqw.eq(StringUtils.isNotBlank(bo.getAuditRemark()), ZjkProductSummarizeLog::getAuditRemark, bo.getAuditRemark());
        lqw.eq(bo.getAuditBy() != null, ZjkProductSummarizeLog::getAuditBy, bo.getAuditBy());
        lqw.eq(bo.getAuditTime() != null, ZjkProductSummarizeLog::getAuditTime, bo.getAuditTime());
        return lqw;
    }

    /**
     * 新增总结日志表
     *
     * @param bo 总结日志表
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkProductSummarizeLogBo bo) {
        ZjkProductSummarizeLog add = MapstructUtils.convert(bo, ZjkProductSummarizeLog.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改总结日志表
     *
     * @param bo 总结日志表
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkProductSummarizeLogBo bo) {
        ZjkProductSummarizeLog update = MapstructUtils.convert(bo, ZjkProductSummarizeLog.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkProductSummarizeLog entity) {
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除总结日志表信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if (isValid) {
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public int audit(AuditVo bo) {
        if (bo == null) {
            throw new ServiceException("无效参数");
        }
        Long id = bo.getId();
        if (id == null) {
            throw new ServiceException("无效id");
        }

        String auditStatus = bo.getAuditStatus();
        SummarizeStatusEnum byCode = SummarizeStatusEnum.getByCode(auditStatus);
        if (byCode == null) {
            throw new ServiceException("无效状态");
        }
        ZjkProductSummarizeLog zpsl = baseMapper.selectById(id);
        if (zpsl == null) {
            return 0;
        }
        Long userId = LoginHelper.getUserId();
        zpsl.setAuditBy(userId);
        zpsl.setAuditStatus(auditStatus);
        zpsl.setAuditTime(new Date());
        String auditRemark = bo.getAuditRemark();
        zpsl.setAuditRemark(auditRemark);
        zpsl.setStatus(auditStatus);
        int update = baseMapper.updateById(zpsl);
        bo.setId(zpsl.getSummarizeId());

        // 审批流程记录
        zjkApprovalProcessLogsService.saveCommonApprovalRecord(bo.getAuditStatus().equals(SummarizeStatusEnum.to_be_evaluated.getCode())? ApprovalProcessTypeEnum.APPROVAL_PASSED.getCode():ApprovalProcessTypeEnum.APPROVAL_REFUSED.getCode(),
            ApprovalProcessBusinessTypeEnum.PROJECT_APPROVALS.getCode(),
            bo.getAuditRemark());

        return summarizeService.auditSny(bo);
    }
}
