package org.dromara.resource.controller;


import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import cn.hutool.core.util.ObjectUtil;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.QueryGroup;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.resource.domain.bo.SysOssBo;
import org.dromara.resource.domain.vo.SysOssUploadVo;
import org.dromara.resource.domain.vo.SysOssVo;
import org.dromara.resource.service.ISysOssService;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

/**
 * 文件上传 控制层
 *
 * @author Lion Li
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/oss")
public class SysOssController extends BaseController {

    private final ISysOssService iSysOssService;

    /**
     * 查询OSS对象存储列表
     */
    @GetMapping("/list")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public TableDataInfo<SysOssVo> list(@Validated(QueryGroup.class) SysOssBo bo, PageQuery pageQuery) {
        return iSysOssService.queryPageList(bo, pageQuery);
    }

    /**
     * 查询OSS对象基于id串
     *
     * @param ossIds OSS对象ID串
     */
    @GetMapping("/listByIds/{ossIds}")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public R<List<SysOssVo>> listByIds(@NotEmpty(message = "主键不能为空") @PathVariable Long[] ossIds) {
        if (ObjectUtil.isEmpty(ossIds)) {
            return R.ok();
        }
        List<SysOssVo> list = iSysOssService.listByIds(Arrays.asList(ossIds));
        return R.ok(list);
    }

    /**
     * 上传OSS对象存储
     *
     * @param file 文件
     */
    @Log(title = "OSS对象存储", businessType = BusinessType.INSERT)
    @PostMapping(value = "/upload", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public R<SysOssUploadVo> upload(@RequestPart("file") MultipartFile file) {
        try {
            if (ObjectUtil.isNull(file)|| file.getBytes().length==0) {
                return R.fail("上传文件不能为空");
            }
        } catch (IOException e) {
            return R.fail("上传文件不能为空");
        }

        SysOssVo oss = iSysOssService.upload(file);
        SysOssUploadVo uploadVo = new SysOssUploadVo();
        uploadVo.setUrl(oss.getUrl());
        uploadVo.setFileName(oss.getOriginalName());
        uploadVo.setOssId(oss.getOssId().toString());
        return R.ok(uploadVo);
    }

    /**
     * 下载OSS对象存储
     *
     * @param ossId OSS对象ID
     */
    @GetMapping("/download/{ossId}")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public void download(@PathVariable Long ossId, HttpServletResponse response) throws IOException {
        iSysOssService.download(ossId, response);
    }

    /**
     * 删除OSS对象存储
     *
     * @param ossIds OSS对象ID串
     */
    @Log(title = "OSS对象存储", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ossIds}")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public R<Void> remove(@NotEmpty(message = "主键不能为空") @PathVariable Long[] ossIds) {
        return toAjax(iSysOssService.deleteWithValidByIds(Arrays.asList(ossIds), true));
    }
    /**
     * 下载多个文件为zip
     *
     * @param ossIds OSS对象ID串
     */
    @Log(title = "下载多个文件为zip", businessType = BusinessType.EXPORT)
    @GetMapping("/downloadToZip/{ossIds}")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    public void downloadToZip(@NotEmpty(message = "主键不能为空") @PathVariable List<Long> ossIds,HttpServletResponse response) throws IOException {
         iSysOssService.downloadToZip(ossIds, response);
    }

    /**
     * 查看OSS对象
     */
    @GetMapping("/view/{y}/{m}/{d}/{key}")
    public void view(@PathVariable String y, @PathVariable String m,@PathVariable String d,@PathVariable String key,HttpServletResponse response) throws IOException {
        String fileName = y + "/" + m + "/" + d + "/" + key;
        iSysOssService.view(fileName, response);
    }
}
