package org.dromara.zjk.controller;

import com.alibaba.fastjson.JSON;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.DubboReference;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.system.api.RemoteConfigService;
import org.dromara.zjk.domain.bo.ZjkExpertBo;
import org.dromara.zjk.domain.bo.ZjkProductBo;
import org.dromara.zjk.domain.vo.ProductPanoramaDetailVo;
import org.dromara.zjk.domain.vo.ZjkExpertVo;
import org.dromara.zjk.domain.vo.ZjkProductVo;
import org.dromara.zjk.service.IZjkExpertService;
import org.dromara.zjk.service.IZjkProductService;
import org.dromara.zjk.service.ZjkExpertMoneyInfoService;
import org.dromara.zjk.utils.ApiThreadLocal;
import org.dromara.zjk.utils.EncryptionUtil;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.dromara.zjk.zwy.utils.StringCryptoUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 专家库对外api接口
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/api")
@Slf4j
public class ApiController {
    // 默认的密钥常量
    private static final String DEFAULT_SECRET_KEY = "k6psy7kafT%UaPd2";

    @DubboReference(timeout = 5000)
    private final RemoteConfigService remoteConfigService;

    private final IZjkExpertService zjkExpertService;
    @Autowired
    private StringCryptoUtil stringCryptoUtil;
    private final IZjkProductService zjkProductService;
    private final ZjkExpertMoneyInfoService zjkExpertMoneyInfoService;

    /**
     * 专家库数据列表获取
     *
     * @return
     */
    @DecryptOperation
    @GetMapping("/expertList")
    public R<String> getExpertListData(ZjkExpertBo bo) {
        try {
            PageQuery pageQuery = new PageQuery();
            pageQuery.setPageNum(1);
            pageQuery.setPageSize(2000);
            if (StringUtils.isNotBlank(bo.getContact())) {
                String phone = stringCryptoUtil.encryptField(bo.getContact());
                bo.setContact(phone);
            }
            if (StringUtils.isNotBlank(bo.getIdCard())) {
                String idcard = stringCryptoUtil.encryptField(bo.getIdCard());
                bo.setIdCard(idcard);
            }
            TableDataInfo<ZjkExpertVo> zjkExpertVoTableDataInfo = zjkExpertService.expertList(bo, pageQuery);
            List<ZjkExpertVo> rows = zjkExpertVoTableDataInfo.getRows();
            if (CollectionUtils.isEmpty(rows)) {
                return R.ok("无可返回的数据");
            }
            String ak = ApiThreadLocal.getAk();
            String data = JSON.toJSONString(rows);
            return R.ok("操作成功", data);
        } catch (Exception e) {
            log.error("Unexpected error occurred", e);
            return R.fail("系统异常，请联系管理员");
        }
    }

    /**
     * 专家详情信息获取
     *
     * @return
     */
    @DecryptOperation
    @GetMapping("/expertInfo/{id}")
    public R<String> getExpertInfo(@PathVariable Long id) {
        ZjkExpertVo zjkExpertVo = zjkExpertService.queryById(id);
        if (zjkExpertVo == null) {
            return R.fail("数据不存在");
        }
        String data = JSON.toJSONString(zjkExpertVo);
        return R.ok("操作成功", data);
    }

    /**
     * 项目列表数据获取
     *
     * @return
     */
    @DecryptOperation
    @GetMapping("/productList")
    public R<String> getProductList(ZjkProductBo bo) {
        List<ZjkProductVo> rows = zjkProductService.queryListForApi(bo);
        if (CollectionUtils.isEmpty(rows)) {
            return R.ok("无可返回的数据");
        }
        String data = JSON.toJSONString(rows);
        return R.ok("操作成功", data);
    }

    /**
     * 项目基本细信息获取
     *
     * @return
     */
    @DecryptOperation
    @GetMapping("/productInfo/{id}")
    public R<String> getProductInfo(@PathVariable Long id) {
        ZjkProductVo vo = zjkProductService.queryById(id);
        if (vo == null) {
            return R.fail("数据不存在");
        }
        String data = JSON.toJSONString(vo);
        return R.ok("操作成功", data);
    }

    /**
     * 项目阶段全景细信息获取
     *
     * @return
     */
    @DecryptOperation
    @GetMapping("/productLife/{id}")
    public R<String> getProductLife(@PathVariable Long id) {
        List<ProductPanoramaDetailVo> vo = zjkProductService.getProductPanoramaDetailVoById(id);
        if (vo.size() == 0) {
            return R.fail("数据不存在");
        }
        String data = JSON.toJSONString(vo);
        return R.ok("操作成功", data);
    }

    /**
     * 获取项目某个阶段参评的专家信息
     *
     * @param productId
     * @param reviewSn
     * @return
     */
    @DecryptOperation
    @GetMapping("/stageExperts")
    public R<String> getStageExperts(@NotNull(message = "项目不能为空")
                                     Long productId, @NotNull(message = "阶段不能为空")
                                     Integer reviewSn) {
        List<ZjkExpertVo> res = zjkExpertService.getStageExperts(productId, reviewSn);
        if (CollectionUtils.isEmpty(res)) {
            return R.ok("无可返回的数据");
        }
        String data = JSON.toJSONString(res);
        return R.ok("操作成功", data);
    }

    // 解密接口 (用于调试或者需要解密的地方)
    @PostMapping("/decrypt")
    public String decryptData(
        @RequestHeader("Authorization") String authorizationCode,
        @RequestBody String encryptedData) {
        try {
            // 从配置服务获取密钥
            String SECRET_KEY = remoteConfigService.selectConfigByKey("api.secret.key");
            if (StringUtils.isEmpty(SECRET_KEY)) {
                SECRET_KEY = DEFAULT_SECRET_KEY;
            }

            // 验证授权码
            if (!authorizationCode.equals(SECRET_KEY)) {
                throw new IllegalAccessException("Invalid authorization code");
            }

            // 解密数据
            String decryptedData = EncryptionUtil.decrypt(encryptedData, SECRET_KEY);
            return decryptedData;

        } catch (Exception e) {
            return "Error occurred: " + e.getMessage();
        }
    }


}
