package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkInvoiceBo;
import org.dromara.zjk.domain.vo.InvoiceStatisticsVO;
import org.dromara.zjk.domain.vo.ZjkInvoiceVo;
import org.dromara.zjk.service.IZjkInvoiceService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.dromara.zjk.zwy.annotation.EncryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.ArrayList;
import java.util.List;

/**
 * 发票
 *财务管理》》发票管理
 * @author mzx
 * @date 2024-12-09
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/invoice")
public class ZjkInvoiceController extends BaseController {

    private final IZjkInvoiceService zjkInvoiceService;



    /**
     * 查询发票列表
     *
     */
    @DecryptOperation
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkInvoiceVo> list(ZjkInvoiceBo bo, PageQuery pageQuery) {
        return zjkInvoiceService.queryPageList(bo, pageQuery);
    }

    /**
     * 获取发票详细信息
     *
     * @param id 主键
     */
    @DecryptOperation
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkInvoiceVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkInvoiceService.queryById(id));
    }

    /**
     * 新增发票
     */
    @EncryptOperation
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "发票", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkInvoiceBo bo) {
        return toAjax(zjkInvoiceService.insertByBo(bo));
    }

    /**
     * 修改发票
     */
    @EncryptOperation
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "发票", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkInvoiceBo bo) {
        return toAjax(zjkInvoiceService.updateByBo(bo));
    }
    /**
     * 审核发票
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "审核发票", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @GetMapping("/check/{moneyId}")
    public R<Void> check( @PathVariable  Long moneyId) {
        return toAjax(zjkInvoiceService.check(moneyId));
    }
    /**
     * 删除发票
     *
     * @param ids 主键串
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "发票", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkInvoiceService.deleteWithValidByIds(List.of(ids), true));
    }

    /**
     * 下载excel模板
     *
     * @param
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "发票", businessType = BusinessType.OTHER)
    @PostMapping("/downloadExcel")
    public void downloadExcel(HttpServletResponse response) {
        zjkInvoiceService.downloadExcelTem(response);
    }


    /**
     * 发票信息导入
     * @param file
     * @return
     */
    @Log(title = "发票信息导入", businessType = BusinessType.IMPORT)
    @PostMapping("/imports")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
    }, mode = SaMode.OR)
    public R<Void> imports(@RequestPart("file") MultipartFile file) {
        return zjkInvoiceService.imports(file);
    }



    /**
     * 发票统计和报表
     * @param
     * @return
     */
    @GetMapping("/statistics")
    @SaCheckRole(value = {
            TenantConstants.JG_ROLE_KEY,
    }, mode = SaMode.OR)
    public R<InvoiceStatisticsVO> statistics() {
        InvoiceStatisticsVO invoiceStatisticsVO = zjkInvoiceService.statistics();
        return R.ok(invoiceStatisticsVO);
    }

}
