package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import cn.hutool.core.bean.BeanUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import jakarta.validation.constraints.NotNull;
import org.apache.dubbo.config.annotation.DubboReference;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.system.api.RemoteUserService;
import org.dromara.zjk.domain.ZjkPointReward;
import org.dromara.zjk.domain.ZjkPointRule;
import org.dromara.zjk.domain.ZjkPointUser;
import org.dromara.zjk.domain.dto.PointReqDto;
import org.dromara.zjk.domain.vo.*;
import org.dromara.zjk.service.ZjkPointRewardService;
import org.dromara.zjk.service.ZjkPointRuleService;
import org.dromara.zjk.service.ZjkPointUserService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.springframework.beans.propertyeditors.CustomDateEditor;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

/**
 * 专家库请假控制器
 *
 * @author : yanguohua
 * @version : 1.0
 * @date : 2024/10/30 16:57
 */
@RestController
@RequestMapping("/point")
public class ZjkPointController extends BaseController {
    @Resource
    private ZjkPointRuleService zjkPointRuleService;
    @Resource
    private ZjkPointRewardService zjkPointRewardService;
    @Resource
    private ZjkPointUserService zjkPointUserService;
    @DubboReference
    private  RemoteUserService remoteUserService;
    @InitBinder  //解决前段传过来时间的字符串解析成时间报错问题
    public void initBinder(WebDataBinder webDataBinder){
        webDataBinder.registerCustomEditor(Date.class,new CustomDateEditor(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"),true));
    }
    /**
     * 查询列表-积分规则定义
     */
    @GetMapping("/ruleList")
    @SaCheckPermission("points:rule:index")
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    public R<List<ZjkPointRule>> pointRule(PointReqDto dto) {
        QueryWrapper<ZjkPointRule> wq = new QueryWrapper<>();
        wq.lambda().eq(!StringUtils.isEmpty(dto.getStatus()), ZjkPointRule::getStatus, dto.getStatus())
            .like(!StringUtils.isEmpty(dto.getRuleName()), ZjkPointRule::getRuleName, dto.getRuleName())
            .between(dto.getStartDate() != null && dto.getEndDate() != null, ZjkPointRule::getCreateTime,
                dto.getStartDate(), dto.getEndDate());
        return R.ok(zjkPointRuleService.list(wq));
    }

    /**
     * 查询列表-积分奖励
     */
    @GetMapping("/rewardList")
    @SaCheckPermission("points:motivate:page")
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    public R<List<ZjkPointRewardVO>> rewardList(PointReqDto dto) {
        QueryWrapper<ZjkPointReward> wq = new QueryWrapper<>();
        wq.lambda().eq(!StringUtils.isEmpty(dto.getStatus()), ZjkPointReward::getStatus, dto.getStatus())
            .like(!StringUtils.isEmpty(dto.getRewardName()), ZjkPointReward::getRewardName, dto.getRewardName())
            .between(dto.getStartDate() != null && dto.getEndDate() != null, ZjkPointReward::getCreateTime, dto.getStartDate(), dto.getEndDate());
        List<ZjkPointReward> list = zjkPointRewardService.list(wq);
        List<ZjkPointRewardVO> zjkPointRewardVOS = BeanUtil.copyToList(list, ZjkPointRewardVO.class);
        zjkPointRewardVOS.stream().forEach(item->{
            item.setCreateBy(remoteUserService.selectUserNameById(Long.valueOf(item.getCreateBy())));
        });
        return R.ok(zjkPointRewardVOS);
    }


    /**
     * 新增奖励
     */
    @Log(title = "新增奖励", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("addReward")
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    public R<Void> addReward(@RequestBody ZjkPointReward bo) {
        boolean save = zjkPointRewardService.save(bo);
        if (save) {
            return R.ok();
        } else {
            return R.fail();
        }
    }
    /**
     * 奖励修改
     */
    @Log(title = "奖励修改", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("updateReward")
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    public R<Void> updateReward(@RequestBody ZjkPointReward bo) {
        boolean up = zjkPointRewardService.updateById(bo);
        if (up) {
            return R.ok();
        } else {
            return R.fail();
        }
    }
    /**
     * 奖励修改
     */
    @Log(title = "奖励规则状态修改", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("updateRewardStatus")
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    public R<Void> updateRewardStatus(@RequestBody ZjkPointReward bo) {
        UpdateWrapper<ZjkPointReward> wq = new UpdateWrapper<>();
        wq.lambda().eq(ZjkPointReward::getId, bo.getId()).set( ZjkPointReward::getStatus, bo.getStatus());
        boolean update = zjkPointRewardService.update(wq);
        if (update) {
            return R.ok();
        } else {
            return R.fail();

        }
    }
    /**
     * 积分规则修改
     */
    @Log(title = "积分规则修改", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("updateRuleStatus")
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    public R<Void> updateRuleStatus(@RequestBody PointReqDto bo) {
        UpdateWrapper<ZjkPointRule> wq = new UpdateWrapper<>();
        wq.lambda().eq(ZjkPointRule::getId, bo.getId()).set( ZjkPointRule::getStatus, bo.getStatus());
        boolean update = zjkPointRuleService.update(wq);
        if (update) {
            return R.ok();
        } else {
            return R.fail();

        }
    }
    @GetMapping("/deleteReward/{id}")
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    public  R<Void> deleteRecord(@PathVariable String id) {
         zjkPointRewardService.removeById(id);
        return R.ok();
    }
    /**
     * 奖励详情
     *
     * @param gid 主键
     */
    @GetMapping("/rewardDetail/{gid}")
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    public R<ZjkPointReward> getRewardInfo(@NotNull(message = "主键不能为空")
                                    @PathVariable String gid) {
        return R.ok(zjkPointRewardService.getById(gid));
    }

    /**
     * 专家积分列表
     */
    @DecryptOperation
    @GetMapping("/pointList")
    public TableDataInfo<PointShowVO> pointList(PointReqDto bo, PageQuery pageQuery) {
        return zjkPointUserService.queryPointList(bo, pageQuery);
    }
    /**
     * 积分复核列表
     */
    @GetMapping("/checkPointList")
    @SaCheckRole(value={
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
    },mode = SaMode.OR)
    public TableDataInfo<PointShowVO> checkPointList(PointReqDto bo, PageQuery pageQuery) {
        return zjkPointUserService.checkPointList(bo, pageQuery);
    }
    /**
     * 积分历史记录列表
     */
    @GetMapping("/pointDetail")
    public TableDataInfo<ZjkPointVO> pointDetail(PointReqDto bo, PageQuery pageQuery) {
        return zjkPointUserService.queryPointDetail(bo,pageQuery);
    }
    /**
     * 积分记录删除
     */
    @GetMapping("/deleteRecord/{id}")
    public  R<Void> deleteRecord(@PathVariable Long id,String expertId) {
        return zjkPointUserService.deleteRecord(id,expertId);
    }
    /**
     * 积分趋势分析
     */
    @GetMapping("/pointTrend")
    public  R<PointTrendVO> pointTrend(String dayKey,String expertId) {
        return zjkPointUserService.getPointTrend(dayKey,expertId);
    }
    /**
     * 积分复核接口
     */
    @PostMapping ("/pointCheck")
    public  R<Void> pointCheck(@RequestBody ZjkPointUser dto) {
        return zjkPointUserService.pointCheck(dto);
    }
    /**
     * 积分复核详情
     */
    @GetMapping("/pointCheckDetail")
    public  R<ZjkPointUser> pointCheckDetail(String expertId) {
        QueryWrapper<ZjkPointUser> wq = new QueryWrapper<>();
        wq.lambda().eq(ZjkPointUser::getExpertId,expertId);
        return R.ok(zjkPointUserService.getOne(wq));
    }
    /**
     * 专家积分榜基本信息
     */
    @DecryptOperation
    @GetMapping("/pointBase")
    public  R<PointBaseVO> pointBase() {
        return zjkPointUserService.getPointBase();
    }
    /**
     * 专家积分榜基本信息
     */
    @GetMapping("/expertPointRank")
    public  R<PointExpertRankVO> expertPointRank(Integer rangeDay) {
        return zjkPointUserService.getExpertPointRank(rangeDay);
    }
}
