package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.ProductCountReport;
import org.dromara.zjk.domain.ZjkProduct;
import org.dromara.zjk.domain.bo.ZjkProductBo;
import org.dromara.zjk.domain.dto.ProjectSummaryStatistics;
import org.dromara.zjk.domain.vo.*;
import org.dromara.zjk.service.IZjkProductService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.dromara.zjk.zwy.annotation.EncryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Set;

/**
 * 项目管理
 * 前端访问路由地址为:/zjk/product
 *
 * @author ygh
 * @date 2024-11-01
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/product")
public class ZjkProductController extends BaseController {

    private final IZjkProductService zjkProductService;

    /**
     * 查询项目管理列表
     */
    @DecryptOperation
    @SaCheckPermission("zjk:product:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkProductVo> list(ZjkProductBo bo, PageQuery pageQuery) {
        return zjkProductService.queryPageList(bo, pageQuery);
    }
    @DecryptOperation
    @GetMapping("/get-product")
    public TableDataInfo<ZjkProductVo> getProduct(PageQuery pageQuery,Long userId) {
        return zjkProductService.getProduct(pageQuery, userId);
    }
    @DecryptOperation
    @GetMapping("/get-product-review")
    public List<ZjkReviewProductVo> getProduct_redirectView(Long productId) {
        return zjkProductService.getProductReview(productId);
    }
    @GetMapping("/get-item-id")
    public ItemVO getItemId(Long productId, Long reviewId) {
        return zjkProductService.getItemId(productId, reviewId);
    }


    /**
     * 查询项目管理列表
     */
    @DecryptOperation
    @SaCheckPermission("zjk:product:list")
    @GetMapping("/queryByList")
    public TableDataInfo<ZjkProduct> queryByList() {
        return zjkProductService.queryByList();
    }


    /**
     * 导出项目管理列表
     */
    @DecryptOperation
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目管理", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkProductBo bo, HttpServletResponse response) {
        List<ZjkProductVo> list = zjkProductService.queryList(bo);
        ExcelUtil.exportExcel(list, "项目管理", ZjkProductVo.class, response);
    }

    /**
     * 获取项目管理详细信息
     *
     * @param id 主键
     */
    @DecryptOperation
    @GetMapping("/{id}")
    public R<ZjkProductVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkProductService.queryById(id));
    }

    /**
     * 新增项目管理
     */
    @EncryptOperation
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkProductBo bo) {
        return toAjax(zjkProductService.insertByBo(bo));
    }
    /**
     * 新增项目管理
     */
    @EncryptOperation
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "项目新增审批", businessType = BusinessType.APPROVE)
    @RepeatSubmit()
    @PostMapping("/approval")
    public R<Void> approval(@RequestBody ZjkProductBo bo) {
        return toAjax(zjkProductService.approval(bo));
    }

    /**
     * 修改项目管理
     */
    @EncryptOperation
    @SaCheckPermission("zjk:product:edit")
    @Log(title = "项目管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update")
    public R<Void> edit(@RequestBody ZjkProductBo bo) {
        return toAjax(zjkProductService.updateByBo(bo));
    }

    /**
     * 删除项目管理
     *
     * @param ids 主键串
     */
    @SaCheckPermission("zjk:product:remove")
    @Log(title = "项目管理", businessType = BusinessType.DELETE)
    @GetMapping("delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkProductService.deleteWithValidByIds(List.of(ids), true));
    }

    /**
     * 获取项目全景 部门
     * @return
     */
    @DecryptOperation
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/panorama/dept")
    public R<Set<ProductPanoramaDeptVo>> productPanoramaDept() {
        return R.ok(zjkProductService.productPanoramaDept());
    }

    @DecryptOperation
    @SaCheckPermission("zjk:product:panorama")
    @GetMapping("/panorama")
    public TableDataInfo<ProductPanoramaVo> productPanorama(ProductPanoramaVo vo, PageQuery pageQuery) {
        return zjkProductService.productPanorama(vo,pageQuery);
    }

    @DecryptOperation
    @GetMapping("/panorama-by-id")
    public R<List<ProductPanoramaDetailVo>> getProductPanoramaById(Long productId) {
        return R.ok(zjkProductService.getProductPanoramaDetailVoById(productId));
    }

    /**
     * 获取项目总数
     * @return
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("/getProductPanoramaCount")
    public R<Long> getProductPanoramaCount() {
        return R.ok("success",zjkProductService.getProductPanoramaCount());
    }

    /**
     * 各项目类型数量统计
     * @return
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("getProductTypeCountReport")
    public R<List<ProductCountReport>> getProductTypeCountReport() {
        return R.ok(zjkProductService.getProductTypeCountReport());
    }


    /**
     * 各评审阶段总结数量统计
     * @return
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("/getReviewStageSummary")
    public R<List<ProjectSummaryStatisticsVo>> getReviewStageSummary() {
        return R.ok(zjkProductService.getReviewStageSummary());
    }

    /**
     * 查询项目    项目名称动态查询
     */
    @DecryptOperation
    @GetMapping("/listByName")
    public R<List<ZjkProductVo>> listByName(ZjkProductBo bo) {
        return R.ok(zjkProductService.queryPageListByName(bo));
    }


    /**
     * 新增项目文档
     */
    @SaCheckRole(value = {
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> addBo(@RequestBody ZjkProductBo bo) {
        return toAjax(zjkProductService.addBo(bo));
    }

    /**
     * 修改项目文档
     */
    @SaCheckRole(value = {
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/productReview/update")
    public R<Void> productReviewUpdate(@RequestBody ZjkProductBo bo) {
        return toAjax(zjkProductService.productReviewUpdate(bo));
    }

    /**
     * 查询项目文档 专利
     * @param bo
     * @param pageQuery
     * @return
     */
    @DecryptOperation
    @SaCheckRole(value = {
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目管理", businessType = BusinessType.INSERT)
    @GetMapping("/select/list")
    public TableDataInfo<ZjkProductVo> selectList(ZjkProductBo bo, PageQuery pageQuery) {
        return zjkProductService.selectList(bo,pageQuery);
    }

    /**
     * 获取 项目文档  通过项目id
     * @param productReviewId
     * @return
     */
    @DecryptOperation
    @SaCheckRole(value = {
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目管理", businessType = BusinessType.INSERT)
    @GetMapping("/select/productId/reviewSn/{productReviewId}")
    public R<ZjkProductVo> selectProductId(@PathVariable Long productReviewId) {
        return R.ok(zjkProductService.selectDocumentcByproductId(productReviewId));
    }


    /**
     *   删除 项目文档   项目id
     * @param productReviewId
     * @return
     */
    @SaCheckRole(value = {
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目管理", businessType = BusinessType.DELETE)
    @GetMapping("/remove/productId/reviewSn/{productReviewId}")
    public R removeByProductId(@PathVariable Long productReviewId) {
        return R.ok(zjkProductService.removeByProductId(productReviewId));
    }


    /**
     * 项目总结统计信息
     * */
    @DecryptOperation
    @SaCheckRole(value = {
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "项目管理", businessType = BusinessType.DELETE)
    @GetMapping("/project/summary/statistics")
    public R<ProjectSummaryStatistics> projectSummaryStatistics() {
        return R.ok(zjkProductService.projectSummaryStatistics());
    }

}
