package org.dromara.zjk.expert.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.expert.retirementRecords.domain.bo.ZjkRetirementRecordsBo;
import org.dromara.zjk.expert.retirementRecords.domain.vo.ZjkRetirementRecordsVo;
import org.dromara.zjk.expert.retirementRecords.service.IZjkRetirementRecordsService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.dromara.zjk.zwy.utils.StringCryptoUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 专家退休记录
 * 前端访问路由地址为:/expert/retirementRecords
 *
 * @author zhangzhou
 * @date 2024-11-19
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/retirementRecords")
public class ZjkRetirementRecordsController extends BaseController {

    private final IZjkRetirementRecordsService zjkRetirementRecordsService;

    @Autowired
    private StringCryptoUtil stringCryptoUtil;

    /**
     * 查询专家退休记录列表
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    @DecryptOperation
    @GetMapping("/list")
    public TableDataInfo<ZjkRetirementRecordsVo> list(ZjkRetirementRecordsBo bo, PageQuery pageQuery) {

        // 2025-08-11 搜索时敏感字段需先加密
        String encryptContact = stringCryptoUtil.encryptField(bo.getContact());
        String encryptIdCard = stringCryptoUtil.encryptField(bo.getIdCard());
        bo.setContact(encryptContact);
        bo.setContact(encryptIdCard);

        return zjkRetirementRecordsService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出专家退休记录列表
     */
    @SaCheckPermission("expert:retirementRecords:export")
    @Log(title = "专家退休记录", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkRetirementRecordsBo bo, HttpServletResponse response) {
        List<ZjkRetirementRecordsVo> list = zjkRetirementRecordsService.queryList(bo);
        ExcelUtil.exportExcel(list, "专家退休记录", ZjkRetirementRecordsVo.class, response);
    }

    /**
     * 获取专家退休记录详细信息
     *
     * @param id 主键
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    @SaCheckPermission("expert:retirementRecords:query")
    @DecryptOperation
    @GetMapping("/{id}")
    public R<ZjkRetirementRecordsVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkRetirementRecordsService.queryById(id));
    }

    /**
     * 新增专家退休记录
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    }, mode = SaMode.OR)
    @Log(title = "专家退休记录", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkRetirementRecordsBo bo) {
        return toAjax(zjkRetirementRecordsService.insertByBo(bo));
    }

    /**
     * 修改专家退休记录
     */
    @SaCheckPermission("expert:retirementRecords:edit")
    @Log(title = "专家退休记录", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkRetirementRecordsBo bo) {
        return toAjax(zjkRetirementRecordsService.updateByBo(bo));
    }

    /**
     * 删除专家退休记录
     *
     * @param ids 主键串
     */
    @SaCheckPermission("expert:retirementRecords:remove")
    @Log(title = "专家退休记录", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkRetirementRecordsService.deleteWithValidByIds(List.of(ids), true));
    }
}
