package org.dromara.zjk.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.zjk.domain.ZjkApprovalProcessLogs;
import org.dromara.zjk.domain.bo.ZjkApprovalProcessLogsBo;
import org.dromara.zjk.domain.vo.ZjkApprovalProcessLogsVo;
import org.dromara.zjk.domain.vo.ZjkDataChangesLogsVo;
import org.dromara.zjk.mapper.ZjkApprovalProcessLogsMapper;
import org.dromara.zjk.mapper.ZjkLearningMaterialsMapper;
import org.dromara.zjk.service.IZjkApprovalProcessLogsService;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 审批流程记录Service业务层处理
 *
 * @author zhangzhou
 * @date 2024-11-06
 */
@RequiredArgsConstructor
@Service
public class ZjkApprovalProcessLogsServiceImpl implements IZjkApprovalProcessLogsService {

    private final ZjkApprovalProcessLogsMapper baseMapper;
    private final ZjkLearningMaterialsMapper zjkLearningMaterialsMapper;
    /**
     * 查询审批流程记录
     *
     * @param id 主键
     * @return 审批流程记录
     */
    @Override
    public ZjkApprovalProcessLogsVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询审批流程记录列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 审批流程记录分页列表
     */
    @Override
    public TableDataInfo<ZjkApprovalProcessLogsVo> queryPageList(ZjkApprovalProcessLogsBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkApprovalProcessLogs> lqw = buildQueryWrapper(bo);
        Page<ZjkApprovalProcessLogsVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        result.getRecords().forEach(r -> r.setCreateByStr(zjkLearningMaterialsMapper.selectVoByUserId(r.getCreateBy())));
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的审批流程记录列表
     *
     * @param bo 查询条件
     * @return 审批流程记录列表
     */
    @Override
    public List<ZjkApprovalProcessLogsVo> queryList(ZjkApprovalProcessLogsBo bo) {
        LambdaQueryWrapper<ZjkApprovalProcessLogs> lqw = buildQueryWrapper(bo);
        List<ZjkApprovalProcessLogsVo> zjkDataChangesLogsVos = baseMapper.selectVoList(lqw);
        zjkDataChangesLogsVos.forEach(r -> r.setCreateByStr(zjkLearningMaterialsMapper.selectVoByUserId(r.getCreateBy())));
        return zjkDataChangesLogsVos;
    }

    private LambdaQueryWrapper<ZjkApprovalProcessLogs> buildQueryWrapper(ZjkApprovalProcessLogsBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkApprovalProcessLogs> lqw = Wrappers.lambdaQuery();
        lqw.eq(StringUtils.isNotBlank(bo.getActionItem()), ZjkApprovalProcessLogs::getActionItem, bo.getActionItem());
        lqw.eq(StringUtils.isNotBlank(bo.getActionContent()), ZjkApprovalProcessLogs::getActionContent, bo.getActionContent());
        lqw.between(bo.getStartTime()!=null && bo.getEndTime() != null,
            ZjkApprovalProcessLogs::getCreateTime ,bo.getStartTime(), bo.getEndTime());
        lqw.orderBy(true, false, ZjkApprovalProcessLogs::getCreateTime);
        return lqw;
    }

    /**
     * 新增审批流程记录
     *
     * @param bo 审批流程记录
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkApprovalProcessLogsBo bo) {
        ZjkApprovalProcessLogs add = MapstructUtils.convert(bo, ZjkApprovalProcessLogs.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改审批流程记录
     *
     * @param bo 审批流程记录
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkApprovalProcessLogsBo bo) {
        ZjkApprovalProcessLogs update = MapstructUtils.convert(bo, ZjkApprovalProcessLogs.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkApprovalProcessLogs entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除审批流程记录信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public Boolean saveCommonApprovalRecord(String actionItem, String actionContent,String approvalOpinions) {
        ZjkApprovalProcessLogs zjkApprovalProcessLogs = new ZjkApprovalProcessLogs();
        zjkApprovalProcessLogs.setActionItem(actionItem);
        zjkApprovalProcessLogs.setActionContent(actionContent);
        zjkApprovalProcessLogs.setApprovalOpinions(approvalOpinions);
        return baseMapper.insert(zjkApprovalProcessLogs) > 0;
    }
}
