package org.dromara.zjk.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.service.DictService;
import org.dromara.common.core.utils.DateUtils;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.zjk.domain.ZjkExpertLog;
import org.dromara.zjk.domain.bo.ZjkExpertLogBo;
import org.dromara.zjk.domain.vo.ZjkExpertLogVo;
import org.dromara.zjk.mapper.ZjkExpertLogMapper;
import org.dromara.zjk.service.IZjkExpertLogService;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 入库记录Service业务层处理
 *
 * @author zhangzhou
 * @date 2024-10-31
 */
@RequiredArgsConstructor
@Service
public class ZjkExpertLogServiceImpl implements IZjkExpertLogService {

    private final ZjkExpertLogMapper baseMapper;
    private final DictService dictService;

    /**
     * 查询入库记录
     *
     * @param id 主键
     * @return 入库记录
     */
    @Override
    public ZjkExpertLogVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询入库记录列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 入库记录分页列表
     */
    @Override
    public TableDataInfo<ZjkExpertLogVo> queryPageList(ZjkExpertLogBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkExpertLog> lqw = buildQueryWrapper(bo);
        Page<ZjkExpertLogVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的入库记录列表
     *
     * @param bo 查询条件
     * @return 入库记录列表
     */
    @Override
    public List<ZjkExpertLogVo> queryList(ZjkExpertLogBo bo) {
        LambdaQueryWrapper<ZjkExpertLog> lqw = buildQueryWrapper(bo);
        List<ZjkExpertLogVo> zjkExpertLogVos = baseMapper.selectVoList(lqw);
        zjkExpertLogVos.stream().forEach(item->{
            item.setEntryResultStr(item.getEntryResult()==1?"通过":"入库失败");
        });
        return zjkExpertLogVos;
    }

    private LambdaQueryWrapper<ZjkExpertLog> buildQueryWrapper(ZjkExpertLogBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkExpertLog> lqw = Wrappers.lambdaQuery();
        lqw.like(StringUtils.isNotBlank(bo.getName()), ZjkExpertLog::getName, bo.getName());
        lqw.like(StringUtils.isNotBlank(bo.getOrganization()), ZjkExpertLog::getOrganization, bo.getOrganization());
        lqw.eq(bo.getEntryResult() != null, ZjkExpertLog::getEntryResult, bo.getEntryResult());
        lqw.orderByDesc(ZjkExpertLog::getCreateTime);
        return lqw;
    }

    /**
     * 新增入库记录
     *
     * @param bo 入库记录
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkExpertLogBo bo) {
        ZjkExpertLog add = MapstructUtils.convert(bo, ZjkExpertLog.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改入库记录
     *
     * @param bo 入库记录
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkExpertLogBo bo) {
        ZjkExpertLog update = MapstructUtils.convert(bo, ZjkExpertLog.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkExpertLog entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除入库记录信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public List<Map<String, Object>> getApplicationResults(String startTime, String endTime) {
        if (StringUtils.isBlank(startTime) || StringUtils.isBlank(endTime)) {
            return baseMapper.getApplicationResults(DateUtils.getCurrentMonthDateRange()[0], DateUtils.getCurrentMonthDateRange()[1]);
        } else {
            return baseMapper.getApplicationResults(startTime, endTime);
        }
    }
}
