package org.dromara.zjk.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import jakarta.servlet.http.HttpServletResponse;
import org.dromara.common.core.domain.R;
import org.dromara.common.excel.core.ExcelResult;
import org.dromara.zjk.domain.ZjkExpert;
import org.dromara.zjk.mapper.*;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.excel.core.DropDownOptions;
import lombok.extern.slf4j.Slf4j;
import org.dromara.common.core.exception.ServiceException;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.satoken.utils.LoginHelper;
import org.dromara.zjk.domain.ZjkExpertMoneyInfo;
import org.dromara.zjk.domain.ZjkInvoice;
import org.dromara.zjk.domain.ZjkSettlementVoucher;
import org.dromara.zjk.domain.bo.ZjkInvoiceBo;
import org.dromara.zjk.domain.vo.*;
import org.dromara.zjk.enums.MoneySettlementStatusEnum;
import org.dromara.zjk.mapper.ZjkExpertMoneyConfigMapper;
import org.dromara.zjk.mapper.ZjkExpertMoneyInfoMapper;
import org.dromara.zjk.mapper.ZjkInvoiceMapper;
import org.dromara.zjk.mapper.ZjkSettlementVoucherMapper;
import org.dromara.zjk.service.IZjkInvoiceService;
import org.dromara.zjk.service.ZjkExpertMoneyInfoService;
import org.dromara.zjk.zwy.client.HmacClient;
import org.dromara.zjk.zwy.utils.ObjectHashGenerator;
import org.dromara.zjk.zwy.utils.StringCryptoUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;
import org.dromara.zjk.enums.ReviewSnEnum;


/**
 * 发票Service业务层处理
 *
 * @author mzx
 * @date 2024-12-09
 *
 */
@Slf4j
@RequiredArgsConstructor
@Service
public class ZjkInvoiceServiceImpl implements IZjkInvoiceService {
    @Resource
    private final ZjkInvoiceMapper zjkInvoiceMapper;
    @Resource
    private final ZjkProductMapper zjkProductMapper;
    @Resource
    private final ZjkSettlementVoucherMapper zjkSettlementVoucherMapper;
    @Resource
    private ZjkExpertMoneyInfoMapper zjkExpertMoneyInfoMapper;
    @Autowired
    private ZjkExpertMoneyConfigMapper zjkExpertMoneyConfigMapper;
    @Resource
    private ZjkExpertMapper zjkExpertMapper;
    private final HmacClient hmacClient;
    @Autowired
    private StringCryptoUtil stringCryptoUtil;
    @Resource
    private final ZjkExpertMoneyInfoService zjkExpertMoneyInfoService;
    @Resource
    private final ZjkReviewPhaseMapper zjkReviewPhaseMapper;
    /**
     * 查询发票
     *
     * @param id 主键
     * @return 发票
     */
    @Override
    public ZjkInvoiceVo queryById(Long id) {
        return zjkInvoiceMapper.selectInfoById(id);
    }

    /**
     * 分页查询发票列表
     *
     * @param bo        查询条件  参数职称等级  联系电话 身份证号 专家名称
     * @param pageQuery 分页参数
     * @return 发票分页列表
     */
    @Override
    public TableDataInfo<ZjkInvoiceVo> queryPageList(ZjkInvoiceBo bo, PageQuery pageQuery) {
        String idcard = stringCryptoUtil.encryptField(bo.getExpertIdNumber());
        String phone = stringCryptoUtil.encryptField(bo.getExpertPhone());
        bo.setExpertPhone(phone);
        bo.setExpertIdNumber(idcard);
        Page<ZjkInvoiceVo> result = zjkInvoiceMapper.queryPageList(pageQuery.build(),bo);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的发票列表
     *
     * @param bo 查询条件
     * @return 发票列表
     */
    @Override
    public List<ZjkInvoiceVo> queryList(ZjkInvoiceBo bo) {
        LambdaQueryWrapper<ZjkInvoice> lqw = buildQueryWrapper(bo);
        return zjkInvoiceMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkInvoice> buildQueryWrapper(ZjkInvoiceBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkInvoice> lqw = Wrappers.lambdaQuery();
        lqw.eq(bo.getInvoiceNumber() != null, ZjkInvoice::getInvoiceNumber, bo.getInvoiceNumber());
        lqw.eq(bo.getInvoiceMoney() != null, ZjkInvoice::getInvoiceMoney, bo.getInvoiceMoney());
        lqw.eq(bo.getStartDate() != null, ZjkInvoice::getStartDate, bo.getStartDate());
        lqw.like(StringUtils.isNotBlank(bo.getExpertName()), ZjkInvoice::getExpertName, bo.getExpertName());
        lqw.eq(StringUtils.isNotBlank(bo.getExpertPhone()), ZjkInvoice::getExpertPhone, bo.getExpertPhone());
        lqw.eq(StringUtils.isNotBlank(bo.getExpertIdNumber()), ZjkInvoice::getExpertIdNumber, bo.getExpertIdNumber());
        lqw.eq(StringUtils.isNotBlank(bo.getExpertTitleLevel()), ZjkInvoice::getExpertTitleLevel, bo.getExpertTitleLevel());
        lqw.eq(StringUtils.isNotBlank(bo.getInvoiceAttached()), ZjkInvoice::getInvoiceAttached, bo.getInvoiceAttached());
        return lqw;
    }

    /**
     * 新增发票
     *
     * @param bo 发票
     * @return 是否新增成功
     */
/*    @Override
    public Boolean insertByBo(ZjkInvoiceBo bo) {
        ZjkInvoice add = MapstructUtils.convert(bo, ZjkInvoice.class);
        validEntityBeforeSave(add);
        add.setCreateBy(LoginHelper.getLoginUser().getUserId());
        add.setCreateTime(new Date());
        add.setCreateDept(LoginHelper.getLoginUser().getDeptId());
        add.setUpdateBy(LoginHelper.getLoginUser().getUserId());
        add.setUpdateTime(new Date());
        boolean flag = zjkInvoiceMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }*/
    @Override
    public Boolean insertByBo(ZjkInvoiceBo bo) {
        ZjkInvoice add = MapstructUtils.convert(bo, ZjkInvoice.class);
        validEntityBeforeSave(add);

        Long userId = LoginHelper.getUserId();
        Long deptId = LoginHelper.getDeptId();
        Date now = new Date();

        // 设置创建/修改人信息
        add.setCreateBy(userId);
        add.setCreateTime(now);
        add.setCreateDept(deptId);
        add.setUpdateBy(userId);
        add.setUpdateTime(now);

        // 插入
        boolean flag = zjkInvoiceMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());

            // 可选：生成 HMAC
            try {
                ZjkInvoiceVo zjkInvoice = zjkInvoiceMapper.selectInfoById(add.getId());
//                ZjkInvoiceVo entity = MapstructUtils.convert(zjkInvoice, ZjkInvoiceVo.class);
                String hash = ObjectHashGenerator.generateHash(zjkInvoice);
                String base64 = Base64.getEncoder().encodeToString(hash.getBytes(StandardCharsets.UTF_8));
                 String hmac = hmacClient.calculateHmac(base64);
//                String hmac = base64;
                add.setHmac(hmac);
                zjkInvoiceMapper.updateById(add); // 回写 HMAC
            } catch (Exception e) {
                log.warn("发票 HMAC 生成失败，ID: {}", add.getId(), e);
            }
        }

        return flag;
    }

    /**
     * 修改发票
     *
     * @param bo 发票
     * @return 是否修改成功
     */
   /* @Override
    public Boolean updateByBo(ZjkInvoiceBo bo) {
        ZjkInvoice update = MapstructUtils.convert(bo, ZjkInvoice.class);
        update.setUpdateBy(LoginHelper.getLoginUser().getUserId());
        update.setUpdateTime(new Date());
        validEntityBeforeSave(update);
        return zjkInvoiceMapper.updateById(update) > 0;
    }*/
    @Override
    public Boolean updateByBo(ZjkInvoiceBo bo) {
        // 转换对象
        ZjkInvoice update = MapstructUtils.convert(bo, ZjkInvoice.class);
        update.setUpdateBy(LoginHelper.getLoginUser().getUserId());
        update.setUpdateTime(new Date());

        // 前置校验
        validEntityBeforeSave(update);

        // 更新主表
        boolean flag = zjkInvoiceMapper.updateById(update) > 0;
        if (flag) {
            try {
                // 重新查询一遍数据（确保包含自动填充字段）
                ZjkInvoiceVo zjkInvoice = zjkInvoiceMapper.selectInfoById(update.getId());
//                ZjkInvoice entity = MapstructUtils.convert(zjkInvoice, ZjkInvoice.class);
                String hash = ObjectHashGenerator.generateHash(zjkInvoice);
                // 生成哈希和 Base64
                String base64 = Base64.getEncoder().encodeToString(hash.getBytes(StandardCharsets.UTF_8));

                // 调用 HMAC 服务（示例中直接用 base64 代替）
                 String hmac = hmacClient.calculateHmac(base64);
//                String hmac = base64;

                update.setHmac(hmac);
                zjkInvoiceMapper.updateById(update); // 二次更新，仅更新 HMAC 字段

                log.info("发票更新成功，HMAC 已刷新，ID: {}", update.getId());
            } catch (Exception e) {
                log.error("更新发票 HMAC 失败，ID: {}", update.getId(), e);
            }
        }

        return flag;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkInvoice entity){
        //TODO 做一些数据校验,如唯一约束
        //发票号码验证是否已存在
        String invoiceNumber = entity.getInvoiceNumber();
        LambdaQueryWrapper<ZjkInvoice> zjkInvoiceLambdaQueryWrapper = new LambdaQueryWrapper<>();
        zjkInvoiceLambdaQueryWrapper.eq(ZjkInvoice::getInvoiceNumber, invoiceNumber);
        if (entity.getId()!=null){
            zjkInvoiceLambdaQueryWrapper.ne(ZjkInvoice::getId, entity.getId());
        }
        Long invoiceCount = zjkInvoiceMapper.selectCount(zjkInvoiceLambdaQueryWrapper);
        log.info("invoiceCount:{}", invoiceCount);
        if (invoiceCount > 0){
            throw new ServiceException("已存在发票号为:"+invoiceNumber+"的发票");
        }
    }

    /**
     * 校验并批量删除发票信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return zjkInvoiceMapper.deleteByIds(ids) > 0;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public R<Void> imports(MultipartFile file) {
        try {
            InputStream inputStream = file.getInputStream();
            // 文件转化
            ExcelResult<ZjkInvoice> zjkInvoiceExcelResult = ExcelUtil.importExcel(inputStream, ZjkInvoice.class, true);
            List<ZjkInvoice> zjkInvoiceList = zjkInvoiceExcelResult.getList();
            if (zjkInvoiceList != null && !zjkInvoiceList.isEmpty()) {
                //专家姓名转换为专家id
                StringBuffer errorSb = new StringBuffer();
                zjkInvoiceList.stream().forEach(item->{
                    String ename = item.getExpertName();
                    item.setReviewSn(Integer.valueOf(item.getReviewSnStr()));
                    //首先校验项目 阶段 专家 结算单号  四者条件下是否存在该结算单，存在时才能导入
                    ZjkExpertMoneyInfo moneyInfo = zjkInvoiceMapper.checkIfImport(item.getProductName(), item.getReviewSn(), item.getExpertName(), item.getSettleNo());
                    if(moneyInfo==null){
                        String msg=ename+"所参与的"+item.getProductName()+ReviewSnEnum.getDescByCode(String.valueOf(item.getReviewSn()))+"结算单号为："+item.getSettleNo()+"的结算单不存在";
                        errorSb.append( msg);
                        errorSb.append("<br/>");
                    }else{
                        LambdaQueryWrapper<ZjkExpert> wq = new LambdaQueryWrapper<>();
                        if(StringUtils.isNotBlank(item.getExpertIdNumber())){
                            String idcard = stringCryptoUtil.encryptField(item.getExpertIdNumber());
                            item.setExpertIdNumber(idcard);
                            wq.eq(ZjkExpert::getIdCard, idcard);
                        }
                        if(StringUtils.isNotBlank(item.getExpertName())){
                            wq.eq(ZjkExpert::getExpertName, item.getExpertName());
                            ZjkExpert zjkExpert = zjkExpertMapper.selectOne(wq);
                            if(zjkExpert==null){
                                String msg="身份证为："+item.getExpertIdNumber()+",专家名为"+item.getExpertName()+"的专家信息不在专家库中！";
                                errorSb.append( msg);
                                errorSb.append("<br/>");
                            }else{
                                Long eid = zjkExpert.getExpertId();
                                item.setExpertName(String.valueOf(eid));
                                List<ZjkExpertVo> eps = zjkExpertMoneyInfoMapper.getExpertListForSettle(Long.valueOf(moneyInfo.getProductId()), item.getReviewSn());
                                List<Long> collect = eps.stream().map(epsItem -> epsItem.getExpertId()).collect(Collectors.toList());
                                if(!collect.contains(eid)){
                                    String msg=ename+"所参与的"+item.getProductName()+ReviewSnEnum.getDescByCode(String.valueOf(item.getReviewSn()))+"结算单号为："+item.getSettleNo()+"已经进行了开票，不可重复开票！";
                                    errorSb.append( msg);
                                    errorSb.append("<br/>");
                                }else{
                                    item.setProductId(Long.valueOf(moneyInfo.getProductId()));
                                    item.setSettleId(moneyInfo.getMoneyId());
                                }

                            }
                        }
                    }

                    if(StringUtils.isNotBlank(item.getExpertPhone())){
                        String phone = stringCryptoUtil.encryptField(item.getExpertPhone());
                        item.setExpertPhone(phone);
                    }


                });

                Set<String> invoiceNumberList = zjkInvoiceList.stream().map(ZjkInvoice::getInvoiceNumber).collect(Collectors.toSet());;
                List<ZjkInvoice> exitsZjkInvoice = zjkInvoiceMapper.selectList(new LambdaQueryWrapper<ZjkInvoice>().in(ZjkInvoice::getInvoiceNumber,invoiceNumberList));
                if (CollUtil.isNotEmpty(exitsZjkInvoice)) {
                    String existingInvoiceNumbersStr = exitsZjkInvoice.stream()
                        .map(ZjkInvoice::getInvoiceNumber)
                        .map(String::valueOf)
                        .collect(Collectors.joining(","));
                    errorSb.append( "以下发票号已存在: " + existingInvoiceNumbersStr + "，请修改后重新导入");
                    errorSb.append("<br/>");
                }
                //处理职称等级
                //得到职称-数字一一对应list
                List<SelectOptionVo> expertMoneyConfigList = zjkExpertMoneyConfigMapper.getExpertMoneyConfig();
                if (expertMoneyConfigList != null && !expertMoneyConfigList.isEmpty()) {
                    for (ZjkInvoice zjkInvoice : zjkInvoiceList) {
                        //因为数据库职称是数字  转换成汉字
                        for (SelectOptionVo selectOptionVo : expertMoneyConfigList) {
                            if (selectOptionVo.getLabel().equals(zjkInvoice.getExpertTitleLevel())) {
                                zjkInvoice.setExpertTitleLevel(selectOptionVo.getValue());//赋予汉字职称
                            }
                        }
                    }
                }
                if(errorSb.length()>0){
                    return R.fail(errorSb.toString());
                }else{
                //insert
                zjkInvoiceMapper.insert(zjkInvoiceList);
                return R.ok();
                }
            }
        }catch (IOException e){
            log.error("导入发票信息失败,{}",e.getMessage());
            throw new ServiceException("导入发票信息失败,请联系管理员");
        }
        return R.ok();
    }

    @Override
    public InvoiceStatisticsVO statistics() {
        InvoiceStatisticsVO invoiceStatisticsVO = new InvoiceStatisticsVO();
        //发票总数
        Long count = zjkInvoiceMapper.selectCount(null);
        invoiceStatisticsVO.setCount(count.intValue());
        //总金额
        BigDecimal totalMoney = zjkInvoiceMapper.selectTotalMoney();
        invoiceStatisticsVO.setTotalMoney(totalMoney);
        //开票单位及其个数  赋值vo
        List<UnitVO> unitVOList = new ArrayList<>();
        /*Set<String> unitSet =  zjkInvoiceMapper.selectAllUnit();
        //遍历得到单位个数
        for (String unit :unitSet){
            LambdaQueryWrapper<ZjkInvoice> zjkInvoiceLambdaQueryWrapper = new LambdaQueryWrapper<>();
            zjkInvoiceLambdaQueryWrapper.eq(ZjkInvoice::getInvoiceUnit, unit);
            Long unitCount = zjkInvoiceMapper.selectCount(zjkInvoiceLambdaQueryWrapper);
            UnitVO unitVO = new UnitVO();
            unitVO.setName(unit);
            unitVO.setCount(unitCount.intValue());
            unitVOList.add(unitVO);
        }
        invoiceStatisticsVO.setUnitList(unitVOList);*/
        return invoiceStatisticsVO;
    }

    @Override
    public Boolean check(Long moneyId) {
        ZjkInvoice zjkInvoice = zjkInvoiceMapper.selectById(moneyId);
        if(zjkInvoice.getInvoiceAttached()==null){
            throw new ServiceException("发票附件不存在，不可审核！");
        }
        zjkInvoice.setCheckStatus(1);
        zjkInvoiceMapper.updateById(zjkInvoice);
        //TODO 向【财务报表和结算凭证】同步该数据
        ZjkSettlementVoucher settlementVoucher = new ZjkSettlementVoucher();
        // 生成报表编号：FP-20250731-YDTJ
        String dateStr = DateUtil.format(new Date(), "yyyyMMdd");
        String pzCode = "pz-" + dateStr;
        settlementVoucher.setVoucherCode(pzCode);
        settlementVoucher.setProductId(zjkInvoice.getProductId());
        settlementVoucher.setExpertId(zjkInvoice.getExpertName()==null?null:Long.valueOf(zjkInvoice.getExpertName()));
        settlementVoucher.setSettleNo(zjkExpertMoneyInfoMapper.selectById(zjkInvoice.getSettleId()).getSettlementSn());
        settlementVoucher.setMoney(zjkInvoice.getInvoiceMoney());
        settlementVoucher.setSyncStatus(1);
        zjkSettlementVoucherMapper.insert(settlementVoucher);
        return true;
    }

    @Override
    public CheckSupportVo getInvByProductInfo(String productId, String reviewSn, String expertId) {

        LambdaQueryWrapper<ZjkExpertMoneyInfo> wq = new LambdaQueryWrapper<>();
        wq.eq(ZjkExpertMoneyInfo::getProductId, productId)
            .eq(ZjkExpertMoneyInfo::getProductType, reviewSn)
            .eq(ZjkExpertMoneyInfo::getExpertId, expertId);
        ZjkExpertMoneyInfo zjkExpertMoneyInfo = zjkExpertMoneyInfoMapper.selectOne(wq);
        CheckSupportVo checkSupportVo = new CheckSupportVo();
        if(zjkExpertMoneyInfo!=null){
            checkSupportVo.setExpertTypeMoneyFee(zjkExpertMoneyInfo.getExpertTypeMoneyFee());
            checkSupportVo.setExpertMoney(zjkExpertMoneyInfo.getExpertMoney());
            //结算后通过结算单号获取发票信息
            if(zjkExpertMoneyInfo.getStatus().equals(MoneySettlementStatusEnum.NO_SETTLEMENT.getCode())){
                LambdaQueryWrapper<ZjkInvoice> invWq = new LambdaQueryWrapper<>();
                invWq.eq(ZjkInvoice::getSettleId, zjkExpertMoneyInfo.getMoneyId()).orderByDesc(ZjkInvoice::getStartDate);
                List<ZjkInvoice> zjkInvoices = zjkInvoiceMapper.selectList(invWq);
                if(!CollectionUtil.isEmpty(zjkInvoices)){
                    //有发票信息，已开票，求出发票总额
                    checkSupportVo.setInvStu("已开票");
                    BigDecimal reduce = zjkInvoices.stream().map(ZjkInvoice::getInvoiceMoney).reduce(BigDecimal.ZERO, BigDecimal::add);
                    checkSupportVo.setInvoiceMoney(reduce);
                    checkSupportVo.setInvFileId(zjkInvoices.get(0).getInvoiceAttached());
                    List<String> fids = zjkInvoices.stream().map(ZjkInvoice::getInvoiceAttached).collect(Collectors.toList());
                    checkSupportVo.setFids(fids);
                }else{
                    checkSupportVo.setInvStu("未开票");
                }
            }
        }else{
            checkSupportVo.setInvStu("未开票");
        }
        return checkSupportVo;
    }

    @Override
    public void downloadExcelTem(HttpServletResponse response) {
//        // 获取项目名称下拉选项数据
//        List<ZjkReviewProductVo> productList = zjkReviewPhaseMapper.getProductList();
//        List<String> productNames = productList.stream().map(ZjkReviewProductVo::getProductName).collect(Collectors.toList());
//       //获取可开票的结算单号list
//        LambdaQueryWrapper<ZjkExpertMoneyInfo> wq = new LambdaQueryWrapper<>();
//        wq.eq(ZjkExpertMoneyInfo::getStatus, MoneySettlementStatusEnum.NO_SETTLEMENT.getCode());
//
//        List<ZjkExpertMoneyInfo> moneyInfoList = zjkExpertMoneyInfoMapper.selectList(wq);
//        List<String> settleSnList = moneyInfoList.stream().map(ZjkExpertMoneyInfo::getSettlementSn).collect(Collectors.toList());
//        //职称等级
        List<SelectOptionVo> expertMoneyConfigList = zjkExpertMoneyConfigMapper.getExpertMoneyConfig();
        List<String> titleList = expertMoneyConfigList.stream().map(SelectOptionVo::getLabel).collect(Collectors.toList());
        // 创建下拉选项
        List<DropDownOptions> options = new ArrayList<>();
//        DropDownOptions settleSnOption = new DropDownOptions(
//            9, // 假设"项目名称"列在第5列（索引为4），需要根据实际列位置调整
//            settleSnList
//        );
//        // 为"项目名称"列添加下拉框选项
//        // 需要确定"项目名称"列在Excel中的索引位置（从0开始）
//        DropDownOptions productNameOption = new DropDownOptions(
//            3, // 假设"项目名称"列在第5列（索引为4），需要根据实际列位置调整
//            productNames
//        );

        DropDownOptions titleOption = new DropDownOptions(
            8,
            titleList
        );
//        options.add(productNameOption);
//        options.add(settleSnOption);
        options.add(titleOption);
        // 创建空列表用于模板
        List<ZjkInvoice> list = new ArrayList<>();

        // 导出带下拉框的Excel模板
        ExcelUtil.exportExcel(list, "发票导入模板", ZjkInvoice.class, response, options);
    }
}
