package org.dromara.zjk.controller;

import java.util.List;

import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.zjk.domain.bo.ZjkCreditRuleBo;
import org.dromara.zjk.domain.bo.ZjkDataChangesLogsBo;
import org.dromara.zjk.domain.vo.ZjkCreditRuleVo;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.domain.vo.ZjkCreditRewardVo;
import org.dromara.zjk.domain.bo.ZjkCreditRewardBo;
import org.dromara.zjk.service.IZjkCreditRewardService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 信用奖励
 * 前端访问路由地址为:/zjk/creditReward
 *
 * @author ygh
 * @date 2024-11-11
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/creditReward")
public class ZjkCreditRewardController extends BaseController {

    private final IZjkCreditRewardService zjkCreditRewardService;

    /**
     * 查询信用奖励列表
     */
//    @SaCheckPermission("zjk:creditReward:list")
    @GetMapping("/list")
    public List<ZjkCreditRewardVo> list(String name) {
        return zjkCreditRewardService.queryPageList(name);
    }

    /**
     * 导出信用奖励列表
     */
    @SaCheckPermission("zjk:creditReward:export")
    @Log(title = "信用奖励", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkCreditRewardBo bo, HttpServletResponse response) {
        List<ZjkCreditRewardVo> list = zjkCreditRewardService.queryList(bo);
        ExcelUtil.exportExcel(list, "信用奖励", ZjkCreditRewardVo.class, response);
    }

    /**
     * 获取信用奖励详细信息
     *
     * @param ID 主键
     */
    @SaCheckPermission("zjk:creditReward:query")
    @GetMapping("/{ID}")
    public R<ZjkCreditRewardVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable String ID) {
        return R.ok(zjkCreditRewardService.queryById(ID));
    }

    /**
     * 新增信用奖励
     */
    @SaCheckPermission("zjk:creditReward:add")
    @Log(title = "信用奖励", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkCreditRewardBo bo) {
        return toAjax(zjkCreditRewardService.insertByBo(bo));
    }

    /**
     * 修改信用奖励
     */
    @SaCheckPermission("zjk:creditReward:edit")
    @Log(title = "信用奖励", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkCreditRewardBo bo) {
        return toAjax(zjkCreditRewardService.updateByBo(bo));
    }

    /**
     * 删除信用奖励
     *
     * @param IDs 主键串
     */
    @SaCheckPermission("zjk:creditReward:remove")
    @Log(title = "信用奖励", businessType = BusinessType.DELETE)
    @DeleteMapping("/{IDs}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable String[] IDs) {
        return toAjax(zjkCreditRewardService.deleteWithValidByIds(List.of(IDs), true));
    }

    /*
     *信用规则列表
     */
    @GetMapping("/ruleList")
    public R<List<ZjkCreditRuleVo>> ruleList(ZjkCreditRuleBo bo) {
        return R.ok(zjkCreditRewardService.ruleList(bo));
    }

    /*
     *信用规则列表
     */
    @GetMapping("/queryRewardlist")
    public List<ZjkCreditRewardVo> queryRewardlist(String name) {
        return zjkCreditRewardService.queryPageList(name);
    }
}
