package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkApprovalProcessLogsBo;
import org.dromara.zjk.domain.vo.ZjkApprovalProcessLogsVo;
import org.dromara.zjk.service.IZjkApprovalProcessLogsService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 审批流程记录
 * 前端访问路由地址为:/traceManagement/approvalProcessLogs
 *
 * @author zhangzhou
 * @date 2024-11-06
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/approvalProcessLogs")
public class ZjkApprovalProcessLogsController extends BaseController {

    private final IZjkApprovalProcessLogsService zjkApprovalProcessLogsService;

    /**
     * 查询审批流程记录列表
     */
    @SaCheckPermission("traceManagement:approvalProcessLogs:list")
    @GetMapping("/list")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    public TableDataInfo<ZjkApprovalProcessLogsVo> list(ZjkApprovalProcessLogsBo bo, PageQuery pageQuery) {
        return zjkApprovalProcessLogsService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出审批流程记录列表
     */
    @SaCheckPermission("traceManagement:approvalProcessLogs:export")
    @Log(title = "审批流程记录", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkApprovalProcessLogsBo bo, HttpServletResponse response) {
        List<ZjkApprovalProcessLogsVo> list = zjkApprovalProcessLogsService.queryList(bo);
        ExcelUtil.exportExcel(list, "审批流程记录", ZjkApprovalProcessLogsVo.class, response);
    }

    /**
     * 获取审批流程记录详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("traceManagement:approvalProcessLogs:query")
    @GetMapping("/{id}")
    public R<ZjkApprovalProcessLogsVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkApprovalProcessLogsService.queryById(id));
    }

    /**
     * 新增审批流程记录
     */
    @SaCheckPermission("traceManagement:approvalProcessLogs:add")
    @Log(title = "审批流程记录", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkApprovalProcessLogsBo bo) {
        return toAjax(zjkApprovalProcessLogsService.insertByBo(bo));
    }

    /**
     * 修改审批流程记录
     */
    @SaCheckPermission("traceManagement:approvalProcessLogs:edit")
    @Log(title = "审批流程记录", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkApprovalProcessLogsBo bo) {
        return toAjax(zjkApprovalProcessLogsService.updateByBo(bo));
    }

    /**
     * 删除审批流程记录
     *
     * @param ids 主键串
     */
    @SaCheckPermission("traceManagement:approvalProcessLogs:remove")
    @Log(title = "审批流程记录", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkApprovalProcessLogsService.deleteWithValidByIds(List.of(ids), true));
    }
}
