package org.dromara.zjk.learning_world.controller;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.validation.constraints.*;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.zjk.learning_world.domain.bo.ZjkLearningRecordsBo;
import org.dromara.zjk.learning_world.domain.vo.ZjkLearningRecordsVo;
import org.dromara.zjk.learning_world.service.IZjkLearningRecordsService;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.log.enums.BusinessType;


/**
 * 学习记录
 * 前端访问路由地址为:/zjk/learningRecords
 *
 * @author wt
 * @date 2024-11-26
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningRecords")
public class ZjkLearningRecordsController extends BaseController {

    private final IZjkLearningRecordsService zjkLearningRecordsService;

    /**
     * 查询学习记录列表
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkLearningRecordsVo> list(ZjkLearningRecordsBo bo, PageQuery pageQuery) {
        return zjkLearningRecordsService.queryPageList(bo, pageQuery);
    }


    /**
     * 新增学习记录
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "学习记录", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> record(@Validated(AddGroup.class) @RequestBody ZjkLearningRecordsBo bo) {
        return toAjax(zjkLearningRecordsService.insertByBo(bo));
    }

    /**
     * 通过课程id 查询学习情况
     *
     * @param  courseId 通过课程id查询
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{courseId}")
    public R<ZjkLearningRecordsVo> getInfoByCourseId(@NotNull(message = "主键不能为空")
                                            @PathVariable Long courseId) {
        return R.ok(zjkLearningRecordsService.getInfoByCourseId(courseId));
    }

    /**
     * 通过id 获取详情
     *
     * @param  id 通过课程id查询
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkLearningRecordsVo> getInfo(@NotNull(message = "主键不能为空")
                                           @PathVariable Long id) {
        return R.ok(zjkLearningRecordsService.queryById(id));
    }

    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/compliance")
    public R<ZjkLearningRecordsVo> compliance() {
        return R.ok(zjkLearningRecordsService.compliance());
    }


}
