### Docker容器化培训文档

#### 1. Docker简介
容器是轻量级、可移植的软件打包技术，核心优势：
- **隔离性**：通过`cgroups`和`namespaces`实现资源隔离
- **一致性**：环境与配置标准化（Dev/Test/Prod环境一致）
- **高效性**：秒级启动，资源利用率比虚拟机高50%以上
- **镜像分层**：增量更新（如基础层`openjdk:11`与应用层分离）

---

#### 2. Dockerfile编制（Java服务示例）
##### 核心步骤：
```dockerfile
# 阶段1：构建应用
FROM maven:3.8.6 AS build
WORKDIR /app
COPY pom.xml .
RUN mvn dependency:go-offline
COPY src ./src
RUN mvn package -DskipTests

# 阶段2：运行环境
FROM openjdk:11-jre-slim
WORKDIR /app
COPY --from=build /app/target/*.jar app.jar
EXPOSE 8080
ENTRYPOINT ["java", "-jar", "app.jar"]
```


##### 关键命令解析：
| 命令 | 作用 | 示例 |
|------|------|------|
| `FROM` | 指定基础镜像 | `FROM openjdk:11` |
| `COPY` | 复制文件 | `COPY target/*.jar app.jar` |
| `RUN` | 执行命令 | `RUN apt-get update` |
| `EXPOSE` | 声明端口 | `EXPOSE 8080` |
| `ENTRYPOINT` | 启动命令 | `ENTRYPOINT ["java", "-jar"]` |
| **多阶段构建** | 减少镜像体积 | 分离构建环境与运行时环境 |

---

#### 3. 镜像操作全流程
##### 3.1 构建镜像
```bash
# 在Dockerfile目录执行（注意末尾的.）
docker build -t my-java-app:1.0 .
```


##### 3.2 验证与运行
```bash
# 查看镜像
docker images

# 运行容器（-d后台运行，-p端口映射）
docker run -d -p 8080:8080 my-java-app:1.0

# 检查日志
docker logs <container_id>
```


##### 3.3 镜像打包与迁移
```bash
# 导出为tar包（用于离线环境）
docker save -o my-app.tar my-java-app:1.0

# 导入镜像
docker load -i my-app.tar

# 推送到镜像仓库
docker tag my-java-app:1.0 registry.example.com/myteam/my-app:1.0
docker push registry.example.com/myteam/my-app:1.0
```


---

#### 最佳实践
1. **镜像优化**：
    - 使用`.dockerignore`排除无关文件
    - 多阶段构建减少最终镜像大小（上例从~600MB降至~150MB）
2. **安全加固**：
    - 非root用户运行：`USER appuser`
    - 定期更新基础镜像
3. **生产建议**：
    - 使用特定版本标签（避免`latest`）
    - 配置健康检查`HEALTHCHECK`

> 附：常用命令速查  
> `docker ps`查看运行容器｜`docker exec -it <id> /bin/bash`进入容器｜`docker system prune`清理资源