package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkDataChangesLogsBo;
import org.dromara.zjk.domain.vo.ZjkDataChangesLogsDetailVo;
import org.dromara.zjk.domain.vo.ZjkDataChangesLogsVo;
import org.dromara.zjk.service.IZjkDataChangesLogsService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 业务数据变更记录
 * 前端访问路由地址为:/dataChangesLogs/dataChangesLogs
 *
 * @author ygh
 * @date 2024-11-07
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/dataChangesLogs")
public class ZjkDataChangesLogsController extends BaseController {

    private final IZjkDataChangesLogsService zjkDataChangesLogsService;

    /**
     * 查询业务数据变更记录列表
     */
    @SaCheckPermission("dataChangesLogs:dataChangesLogs:list")
    @GetMapping("/list")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    public TableDataInfo<ZjkDataChangesLogsVo> list(ZjkDataChangesLogsBo bo, PageQuery pageQuery) {
        return zjkDataChangesLogsService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出业务数据变更记录列表
     */
    @SaCheckPermission("dataChangesLogs:dataChangesLogs:export")
    @Log(title = "业务数据变更记录", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkDataChangesLogsBo bo, HttpServletResponse response) {
        List<ZjkDataChangesLogsVo> list = zjkDataChangesLogsService.queryList(bo);
        ExcelUtil.exportExcel(list, "业务数据变更记录", ZjkDataChangesLogsVo.class, response);
    }

    /**
     * 获取业务数据变更记录详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("dataChangesLogs:dataChangesLogs:query")
    @GetMapping("/{id}")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    public R<ZjkDataChangesLogsDetailVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkDataChangesLogsService.queryById(id));
    }

    /**
     * 新增业务数据变更记录
     */
    @SaCheckPermission("dataChangesLogs:dataChangesLogs:add")
    @Log(title = "业务数据变更记录", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkDataChangesLogsBo bo) {
        return toAjax(zjkDataChangesLogsService.insertByBo(bo));
    }

    /**
     * 修改业务数据变更记录
     */
    @SaCheckPermission("dataChangesLogs:dataChangesLogs:edit")
    @Log(title = "业务数据变更记录", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkDataChangesLogsBo bo) {
        return toAjax(zjkDataChangesLogsService.updateByBo(bo));
    }

    /**
     * 删除业务数据变更记录
     *
     * @param ids 主键串
     */
    @SaCheckPermission("dataChangesLogs:dataChangesLogs:remove")
    @Log(title = "业务数据变更记录", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkDataChangesLogsService.deleteWithValidByIds(List.of(ids), true));
    }
}
