package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkExpertProfessionalBo;
import org.dromara.zjk.domain.vo.ZjkExpertProfessionalVo;
import org.dromara.zjk.service.IZjkExpertProfessionalService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 专家库技术资格关系表
 * 前端访问路由地址为:/expert/expertProfessional
 *
 * @author zhangzhou
 * @date 2024-11-01
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/expertProfessional")
public class ZjkExpertProfessionalController extends BaseController {

    private final IZjkExpertProfessionalService zjkExpertProfessionalService;

    /**
     * 查询专家库技术资格关系表列表
     */
    @SaCheckPermission("expert:expertProfessional:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkExpertProfessionalVo> list(ZjkExpertProfessionalBo bo, PageQuery pageQuery) {
        return zjkExpertProfessionalService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出专家库技术资格关系表列表
     */
    @SaCheckPermission("expert:expertProfessional:export")
    @Log(title = "专家库技术资格关系表", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkExpertProfessionalBo bo, HttpServletResponse response) {
        List<ZjkExpertProfessionalVo> list = zjkExpertProfessionalService.queryList(bo);
        ExcelUtil.exportExcel(list, "专家库技术资格关系表", ZjkExpertProfessionalVo.class, response);
    }

    /**
     * 获取专家库技术资格关系表详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("expert:expertProfessional:query")
    @GetMapping("/{id}")
    public R<ZjkExpertProfessionalVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkExpertProfessionalService.queryById(id));
    }

    /**
     * 新增专家库技术资格关系表
     */
    @SaCheckPermission("expert:expertProfessional:add")
    @Log(title = "专家库技术资格关系表", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkExpertProfessionalBo bo) {
        return toAjax(zjkExpertProfessionalService.insertByBo(bo));
    }

    /**
     * 修改专家库技术资格关系表
     */
    @SaCheckPermission("expert:expertProfessional:edit")
    @Log(title = "专家库技术资格关系表", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkExpertProfessionalBo bo) {
        return toAjax(zjkExpertProfessionalService.updateByBo(bo));
    }

    /**
     * 删除专家库技术资格关系表
     *
     * @param ids 主键串
     */
    @SaCheckPermission("expert:expertProfessional:remove")
    @Log(title = "专家库技术资格关系表", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkExpertProfessionalService.deleteWithValidByIds(List.of(ids), true));
    }

    @SaCheckPermission("expert:expertProfessional:query")
    @GetMapping("/professionalList")
    public R<List<String>> professionalList() {
        return R.ok(zjkExpertProfessionalService.professionalList());
    }
}
