package org.dromara.zjk.expert.expertPaper.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.expert.expertPaper.domain.bo.ZjkExpertPaperBo;
import org.dromara.zjk.expert.expertPaper.domain.vo.ZjkExpertPaperVo;
import org.dromara.zjk.expert.expertPaper.service.IZjkExpertPaperService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 专家论文信息
 * 前端访问路由地址为:/expert/expertPaper
 *
 * @author zhangzhou
 * @date 2024-11-10
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/expertPaper")
public class ZjkExpertPaperController extends BaseController {

    private final IZjkExpertPaperService zjkExpertPaperService;

    /**
     * 查询专家论文信息列表
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkExpertPaperVo> list(ZjkExpertPaperBo bo, PageQuery pageQuery) {
        return zjkExpertPaperService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出专家论文信息列表
     */
    @SaCheckPermission("expert:expertPaper:export")
    @Log(title = "专家论文信息", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkExpertPaperBo bo, HttpServletResponse response) {
        List<ZjkExpertPaperVo> list = zjkExpertPaperService.queryList(bo);
        ExcelUtil.exportExcel(list, "专家论文信息", ZjkExpertPaperVo.class, response);
    }

    /**
     * 获取专家论文信息详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("expert:expertPaper:query")
    @GetMapping("/{id}")
    public R<ZjkExpertPaperVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkExpertPaperService.queryById(id));
    }

    /**
     * 新增专家论文信息
     */
    @SaCheckPermission("expert:expertPaper:add")
    @Log(title = "专家论文信息", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkExpertPaperBo bo) {
        return toAjax(zjkExpertPaperService.insertByBo(bo));
    }

    /**
     * 修改专家论文信息
     */
    @SaCheckPermission("expert:expertPaper:edit")
    @Log(title = "专家论文信息", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkExpertPaperBo bo) {
        return toAjax(zjkExpertPaperService.updateByBo(bo));
    }

    /**
     * 删除专家论文信息
     *
     * @param ids 主键串
     */
    @SaCheckPermission("expert:expertPaper:remove")
    @Log(title = "专家论文信息", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkExpertPaperService.deleteWithValidByIds(List.of(ids), true));
    }

    /**
     *  通过项目id  喝项目阶段 获取
     * @param productId
     * @param reviewSn
     * @return
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/paper/list")
    public R<List<ZjkExpertPaperVo>> paperList(Long productId, String reviewSn) {
        return R.ok(zjkExpertPaperService.paperList(productId, reviewSn));
    }

    /**
     * 获取在库论文总数
     * @return
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("getExpertPaperCount")
    public R<Long> getExpertPaperCount(){
        return R.ok(zjkExpertPaperService.getExpertPaperCount());
    }

}
