package org.dromara.zjk.service.impl;

import cn.hutool.core.io.unit.DataUnit;
import org.dromara.common.core.utils.DateUtils;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.mybatis.core.page.PageQuery;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import lombok.RequiredArgsConstructor;
import org.dromara.zjk.domain.ZjkCreditRule;
import org.dromara.zjk.domain.ZjkCreditUser;
import org.dromara.zjk.domain.ZjkPointReward;
import org.dromara.zjk.domain.bo.ZjkCreditRuleBo;
import org.dromara.zjk.domain.vo.ZjkCreditRuleVo;
import org.dromara.zjk.enums.OperateCodeEnum;
import org.dromara.zjk.mapper.ZjkCreditMapper;
import org.dromara.zjk.mapper.ZjkCreditUserMapper;
import org.springframework.stereotype.Service;
import org.dromara.zjk.domain.bo.ZjkCreditRewardBo;
import org.dromara.zjk.domain.vo.ZjkCreditRewardVo;
import org.dromara.zjk.domain.ZjkCreditReward;
import org.dromara.zjk.mapper.ZjkCreditRewardMapper;
import org.dromara.zjk.service.IZjkCreditRewardService;

import javax.annotation.Resource;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.*;

/**
 * 信用奖励Service业务层处理
 *
 * @author ygh
 * @date 2024-11-11
 */
@RequiredArgsConstructor
@Service
public class ZjkCreditRewardServiceImpl implements IZjkCreditRewardService {

    @Resource
    private final ZjkCreditRewardMapper baseMapper;

    @Resource
    private final ZjkCreditUserMapper zjkCreditUserMapper;

    /**
     * 查询信用奖励
     *
     * @param ID 主键
     * @return 信用奖励
     */
    @Override
    public ZjkCreditRewardVo queryById(String ID){
        return baseMapper.selectVoById(ID);
    }

    /**
     * 分页查询信用奖励列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 信用奖励分页列表
     */
    @Override
    public List<ZjkCreditRewardVo> queryPageList(String name) {
        LambdaQueryWrapper<ZjkCreditReward> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        if(name!=null){
            lambdaQueryWrapper.like(ZjkCreditReward::getRewardName,name);
        }
        List<ZjkCreditRewardVo> zjkCreditRewardVos = baseMapper.selectVoList(lambdaQueryWrapper);
        return zjkCreditRewardVos;
    }

    /**
     * 查询符合条件的信用奖励列表
     *
     * @param bo 查询条件
     * @return 信用奖励列表
     */
    @Override
    public List<ZjkCreditRewardVo> queryList(ZjkCreditRewardBo bo) {
        LambdaQueryWrapper<ZjkCreditReward> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkCreditReward> buildQueryWrapper(ZjkCreditRewardBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkCreditReward> lqw = Wrappers.lambdaQuery();
        lqw.eq(StringUtils.isNotBlank(bo.getID()), ZjkCreditReward::getID, bo.getID());
        lqw.like(StringUtils.isNotBlank(bo.getRewardName()), ZjkCreditReward::getRewardName, bo.getRewardName());
        lqw.eq(bo.getRewardType() != null, ZjkCreditReward::getRewardType, bo.getRewardType());
        lqw.eq(StringUtils.isNotBlank(bo.getRewardContent()), ZjkCreditReward::getRewardContent, bo.getRewardContent());
        lqw.eq(bo.getRewardObj() != null, ZjkCreditReward::getRewardObj, bo.getRewardObj());
        lqw.eq(bo.getEffectFlag() != null, ZjkCreditReward::getEffectFlag, bo.getEffectFlag());
        lqw.eq(bo.getEffectStandard() != null, ZjkCreditReward::getEffectStandard, bo.getEffectStandard());
        lqw.eq(bo.getRewardUntil() != null, ZjkCreditReward::getRewardUntil, bo.getRewardUntil());
        lqw.eq(bo.getIsForever() != null, ZjkCreditReward::getIsForever, bo.getIsForever());
        lqw.eq(bo.getStatus() != null, ZjkCreditReward::getStatus, bo.getStatus());
        lqw.eq(bo.getCreateBy() != null, ZjkCreditReward::getCreateBy, bo.getCreateBy());
        lqw.eq(bo.getCreateTime() != null, ZjkCreditReward::getCreateTime, bo.getCreateTime());
        lqw.eq(bo.getUpdateBy() != null, ZjkCreditReward::getUpdateBy, bo.getUpdateBy());
        lqw.eq(bo.getUpdateTime() != null, ZjkCreditReward::getUpdateTime, bo.getUpdateTime());
        return lqw;
    }

    /**
     * 新增信用奖励
     *
     * @param bo 信用奖励
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkCreditRewardBo bo) {
        ZjkCreditReward add = MapstructUtils.convert(bo, ZjkCreditReward.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setID(add.getID());
        }
        return flag;
    }

    /**
     * 修改信用奖励
     *
     * @param bo 信用奖励
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkCreditRewardBo bo) {
        ZjkCreditReward update = MapstructUtils.convert(bo, ZjkCreditReward.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkCreditReward entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除信用奖励信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<String> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public List<String> getRewardTitle(Long expertId, Long creditCount) {
        List<ZjkCreditReward> reward = baseMapper.getAbleReward();
        ArrayList<String> strings = new ArrayList<>();
        //专家id
        LambdaQueryWrapper<ZjkCreditUser> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        lambdaQueryWrapper.eq(ZjkCreditUser::getExpertId,expertId);
        ZjkCreditUser zjkCreditUser = zjkCreditUserMapper.selectOne(lambdaQueryWrapper);
        if(zjkCreditUser.getRewardTitle()!=null&&zjkCreditUser.getRewardTitle().length()>0&&zjkCreditUser.getRewardTitle().contains(",")){
            String[] split = zjkCreditUser.getRewardTitle().split(",");
            List<String> list = Arrays.asList(split);
            strings.addAll(list);
        }else{
            strings.add(zjkCreditUser.getRewardTitle());
        }
//        reward.stream().forEach(item -> {
//            String result = "";
//            Integer effectFlag = Math.toIntExact(item.getEffectFlag());
//            if (OperateCodeEnum.DAYU.getCode().equals(effectFlag) && creditCount.compareTo(item.getEffectStandard()) >= 0) {
//                strings.add(item.getRewardName());
//            }
//        });
        return strings;
    }

    @Override
    public List<ZjkCreditRuleVo> ruleList(ZjkCreditRuleBo bo) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Date startTime = null;
        if(bo.getStartTime()!=null){
            try {
                startTime = sdf.parse(bo.getStartTime());
            } catch (ParseException e) {
                throw new RuntimeException(e);
            }
        }
        Date endTime = null;
        if(bo.getEndTime()!=null){
            try {
                endTime = sdf.parse(bo.getEndTime());
            } catch (ParseException e) {
                throw new RuntimeException(e);
            }
        }
        List<ZjkCreditRuleVo> zjkCreditRuleVos = baseMapper.selectRuleList(bo.getRuleName(),startTime,endTime);
        return zjkCreditRuleVos;
    }
}
