package org.dromara.zjk.statement.service.impl;

import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.mybatis.core.page.PageQuery;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.dromara.zjk.statement.domain.bo.ZjkMoneyExpertStatementBo;
import org.dromara.zjk.statement.domain.vo.ZjkMoneyExpertStatementVo;
import org.dromara.zjk.statement.domain.ZjkMoneyExpertStatement;
import org.dromara.zjk.statement.mapper.ZjkMoneyExpertStatementMapper;
import org.dromara.zjk.statement.service.IZjkMoneyExpertStatementService;

import java.util.List;
import java.util.Map;
import java.util.Collection;

/**
 * 专家费用报表Service业务层处理
 *
 * @author ygh
 * @date 2024-12-16
 */
@RequiredArgsConstructor
@Service
public class ZjkMoneyExpertStatementServiceImpl implements IZjkMoneyExpertStatementService {

    private final ZjkMoneyExpertStatementMapper baseMapper;

    /**
     * 查询专家费用报表
     *
     * @param id 主键
     * @return 专家费用报表
     */
    @Override
    public ZjkMoneyExpertStatementVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询专家费用报表列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 专家费用报表分页列表
     */
    @Override
    public TableDataInfo<ZjkMoneyExpertStatementVo> queryPageList(ZjkMoneyExpertStatementBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkMoneyExpertStatement> lqw = buildQueryWrapper(bo);
        Page<ZjkMoneyExpertStatementVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的专家费用报表列表
     *
     * @param bo 查询条件
     * @return 专家费用报表列表
     */
    @Override
    public List<ZjkMoneyExpertStatementVo> queryList(ZjkMoneyExpertStatementBo bo) {
        LambdaQueryWrapper<ZjkMoneyExpertStatement> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkMoneyExpertStatement> buildQueryWrapper(ZjkMoneyExpertStatementBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkMoneyExpertStatement> lqw = Wrappers.lambdaQuery();
        lqw.like(StringUtils.isNotBlank(bo.getExpertName()), ZjkMoneyExpertStatement::getExpertName, bo.getExpertName());
        lqw.eq(StringUtils.isNotBlank(bo.getExpertType()), ZjkMoneyExpertStatement::getExpertType, bo.getExpertType());
        lqw.eq(bo.getSumFee() != null, ZjkMoneyExpertStatement::getSumFee, bo.getSumFee());
        lqw.eq(bo.getNoPayFee() != null, ZjkMoneyExpertStatement::getNoPayFee, bo.getNoPayFee());
        lqw.eq(bo.getPayFee() != null, ZjkMoneyExpertStatement::getPayFee, bo.getPayFee());
        lqw.eq(bo.getStatementId() != null, ZjkMoneyExpertStatement::getStatementId, bo.getStatementId());
        lqw.eq(StringUtils.isNotBlank(bo.getMoneyCycle()), ZjkMoneyExpertStatement::getMoneyCycle, bo.getMoneyCycle());
        return lqw;
    }

    /**
     * 新增专家费用报表
     *
     * @param bo 专家费用报表
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkMoneyExpertStatementBo bo) {
        ZjkMoneyExpertStatement add = MapstructUtils.convert(bo, ZjkMoneyExpertStatement.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改专家费用报表
     *
     * @param bo 专家费用报表
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkMoneyExpertStatementBo bo) {
        ZjkMoneyExpertStatement update = MapstructUtils.convert(bo, ZjkMoneyExpertStatement.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkMoneyExpertStatement entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除专家费用报表信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }
}
