package org.dromara.workflow.dubbo;

import lombok.RequiredArgsConstructor;
import org.apache.dubbo.config.annotation.DubboService;
import org.dromara.workflow.api.domain.RemoteWorkflowService;
import org.dromara.workflow.domain.WfTaskBackNode;
import org.dromara.workflow.domain.bo.BackProcessBo;
import org.dromara.workflow.domain.bo.CompleteTaskBo;
import org.dromara.workflow.domain.bo.StartProcessBo;
import org.dromara.workflow.service.*;
import org.dromara.workflow.utils.QueryUtils;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;

/**
 * RemoteWorkflowServiceImpl
 *
 * @Author ZETA
 * @Date 2024/6/3
 */
@DubboService
@RequiredArgsConstructor
public class RemoteWorkflowServiceImpl implements RemoteWorkflowService {

    private final WorkflowService workflowService;
    private final IActHiProcinstService actHiProcinstService;
    private final IActProcessInstanceService iActProcessInstanceService;
    private final IActTaskService iActTaskService;
    private final IWfTaskBackNodeService wfTaskBackNodeService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map<String, Object> startWorkFlow(String businessKey,String tableName,Map<String, Object> variables) {
        StartProcessBo startProcessBo = new StartProcessBo();
        startProcessBo.setBusinessKey(businessKey);
        startProcessBo.setTableName(tableName);
        startProcessBo.setVariables(variables);
        return iActTaskService.startProcessWithBusinessId(startProcessBo);
    }

    @Override
    public Map<String, Object> getTaskInfoByBusinessKey(String businessKey) {
        return QueryUtils.getTaskInfoByBusinessKey(businessKey);
    }

    @Override
    public String getNodeList(String processInstanceId) {
        List<WfTaskBackNode> wfTaskBackNodeList = wfTaskBackNodeService.getListByInstanceId(processInstanceId);
        return wfTaskBackNodeList.get(0).getNodeId();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean completeTask(String taskId, List<String> messageTypeList, Map<String, Object> variables,String businessId) {
        CompleteTaskBo completeTaskBo = new CompleteTaskBo();
        completeTaskBo.setTaskId(taskId);
        completeTaskBo.setMessageType(messageTypeList);
        if (variables != null) {
            completeTaskBo.setVariables(variables);
        }
        completeTaskBo.setBusinessId(businessId);
        return iActTaskService.handleTaskCompletion(completeTaskBo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean cancelProcessApply(String businessKey) {
        return iActProcessInstanceService.handleProcessCancellation(businessKey);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String backProcess(String taskId, List<String> messageTypeList, String targetActivityId, String message) {
        // 构建驳回对象
        BackProcessBo backProcessBo = new BackProcessBo();
        backProcessBo.setTaskId(taskId);
        backProcessBo.setMessageType(messageTypeList);
        backProcessBo.setTargetActivityId(targetActivityId);
        backProcessBo.setMessage(message);
        return iActTaskService.backProcess(backProcessBo);
    }

    @Override
    public boolean deleteRunAndHisInstance(List<String> businessKeys) {
        return workflowService.deleteRunAndHisInstance(businessKeys);
    }

    @Override
    public String getBusinessStatusByTaskId(String taskId) {
        return workflowService.getBusinessStatusByTaskId(taskId);
    }

    @Override
    public String getBusinessStatus(String businessKey) {
        return workflowService.getBusinessStatus(businessKey);
    }

    @Override
    public void setVariable(String taskId, String variableName, Object value) {
        workflowService.setVariable(taskId, variableName, value);
    }

    @Override
    public void setVariables(String taskId, Map<String, Object> variables) {
        workflowService.setVariables(taskId, variables);
    }

    @Override
    public void setVariableLocal(String taskId, String variableName, Object value) {
        workflowService.setVariableLocal(taskId, variableName, value);
    }

    @Override
    public void setVariablesLocal(String taskId, Map<String, Object> variables) {
        workflowService.setVariablesLocal(taskId, variables);
    }

    /**
     * 按照业务id查询流程实例id
     *
     * @param businessKey 业务id
     * @return 结果
     */
    @Override
    public String getInstanceIdByBusinessKey(String businessKey) {
        return workflowService.getInstanceIdByBusinessKey(businessKey);
    }
}
