package org.dromara.zjk.controller;

import java.util.List;

import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.zjk.domain.dto.CreditReqDto;
import org.dromara.zjk.domain.dto.PointReqDto;
import org.dromara.zjk.domain.vo.*;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.domain.bo.ZjkCreditBo;
import org.dromara.zjk.service.IZjkCreditService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 信用记录
 * 前端访问路由地址为:/zjk/credit
 *
 * @author ygh
 * @date 2024-11-11
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/credit")
public class ZjkCreditController extends BaseController {

    private final IZjkCreditService zjkCreditService;

    /**
     * 查询信用记录列表
     */
//    @SaCheckPermission("zjk:credit:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkCreditVo> list(ZjkCreditBo bo, PageQuery pageQuery) {
        return zjkCreditService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出信用记录列表
     */
    @SaCheckPermission("zjk:credit:export")
    @Log(title = "信用记录", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkCreditBo bo, HttpServletResponse response) {
        List<ZjkCreditVo> list = zjkCreditService.queryList(bo);
        ExcelUtil.exportExcel(list, "信用记录", ZjkCreditVo.class, response);
    }

    /**
     * 获取信用记录详细信息
     *
     * @param ID 主键
     */
    @SaCheckPermission("zjk:credit:query")
    @GetMapping("/{ID}")
    public R<ZjkCreditVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long ID) {
        return R.ok(zjkCreditService.queryById(ID));
    }

    /**
     * 新增信用记录
     */
    @SaCheckPermission("zjk:credit:add")
    @Log(title = "信用记录", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkCreditBo bo) {
        return toAjax(zjkCreditService.insertByBo(bo));
    }

    /**
     * 修改信用记录
     */
    @SaCheckPermission("zjk:credit:edit")
    @Log(title = "信用记录", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkCreditBo bo) {
        return toAjax(zjkCreditService.updateByBo(bo));
    }

    /**
     * 删除信用记录
     *
     * @param IDs 主键串
     */
    @SaCheckPermission("zjk:credit:remove")
    @Log(title = "信用记录", businessType = BusinessType.DELETE)
    @DeleteMapping("/{IDs}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] IDs) {
        return toAjax(zjkCreditService.deleteWithValidByIds(List.of(IDs), true));
    }


    /**
     * 同步专家信息到信用用户记录表
     */
    @GetMapping("/expertPointRank/user")
    public  R<PointExpertRankVO> expertPointRankUser() {
        return zjkCreditService.expertPointRankUser();
    }


    /**
     * 专家信用管理列表
     */
    @DecryptOperation
    @GetMapping("/creditList")
    public TableDataInfo<CreditShowVO> creditList(CreditReqDto bo, PageQuery pageQuery) {
        return zjkCreditService.queryPointList(bo, pageQuery);
    }


    /**
     * 专家信用榜基本信息
     */
    @DecryptOperation
    @GetMapping("/pointBase")
    public  R<ChartBaseCreditVO> pointBase() {
        return zjkCreditService.getPointBase();
    }

    //同步到记录表中最初数据
    @GetMapping("/synchronization")
    public  R<ChartBaseCreditVO> synchronization() {
        return zjkCreditService.synchronization();
    }

}
