package org.dromara.zjk.expert.expertAwards.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.expert.expertAwards.domain.bo.ZjkExpertAwardsBo;
import org.dromara.zjk.expert.expertAwards.domain.vo.ZjkExpertAwardsVo;
import org.dromara.zjk.expert.expertAwards.service.IZjkExpertAwardsService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 专家获奖情况
 * 前端访问路由地址为:/expert/expertAwards
 *
 * @author zhangzhou
 * @date 2024-11-10
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/expertAwards")
public class ZjkExpertAwardsController extends BaseController {

    private final IZjkExpertAwardsService zjkExpertAwardsService;

    /**
     * 查询专家获奖情况列表
     */
    @SaCheckPermission("expert:expertAwards:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkExpertAwardsVo> list(ZjkExpertAwardsBo bo, PageQuery pageQuery) {
        return zjkExpertAwardsService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出专家获奖情况列表
     */
    @SaCheckPermission("expert:expertAwards:export")
    @Log(title = "专家获奖情况", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkExpertAwardsBo bo, HttpServletResponse response) {
        List<ZjkExpertAwardsVo> list = zjkExpertAwardsService.queryList(bo);
        ExcelUtil.exportExcel(list, "专家获奖情况", ZjkExpertAwardsVo.class, response);
    }

    /**
     * 获取专家获奖情况详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("expert:expertAwards:query")
    @GetMapping("/{id}")
    public R<ZjkExpertAwardsVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkExpertAwardsService.queryById(id));
    }

    /**
     * 新增专家获奖情况
     */
    @SaCheckPermission("expert:expertAwards:add")
    @Log(title = "专家获奖情况", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkExpertAwardsBo bo) {
        return toAjax(zjkExpertAwardsService.insertByBo(bo));
    }

    /**
     * 修改专家获奖情况
     */
    @SaCheckPermission("expert:expertAwards:edit")
    @Log(title = "专家获奖情况", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkExpertAwardsBo bo) {
        return toAjax(zjkExpertAwardsService.updateByBo(bo));
    }

    /**
     * 删除专家获奖情况
     *
     * @param ids 主键串
     */
    @SaCheckPermission("expert:expertAwards:remove")
    @Log(title = "专家获奖情况", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkExpertAwardsService.deleteWithValidByIds(List.of(ids), true));
    }
}
