package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.InoutCheckBo;
import org.dromara.zjk.domain.vo.CheckSupportVo;
import org.dromara.zjk.domain.vo.InoutCheckVo;
import org.dromara.zjk.domain.vo.ProductPanoramaVo;
import org.dromara.zjk.service.IInoutCheckService;
import org.dromara.zjk.service.IZjkProductService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.util.List;

/**
 * 收支对账
 * 前端访问路由地址为:/zjk/inoutCheck
 *
 * @author ygh
 * @date 2025-07-30
 */
@RequiredArgsConstructor
@RestController
@RequestMapping("/inoutCheck")
public class InoutCheckController extends BaseController {

    private final IInoutCheckService inoutCheckService;
    private final IZjkProductService zjkProductService;

    /**
     * 查询收支对账列表
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<InoutCheckVo> list(InoutCheckBo bo, PageQuery pageQuery) {
        return inoutCheckService.queryPageList(bo, pageQuery);
    }


    /**
     * 获取收支对账详细信息
     *
     * @param ID 主键
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    @GetMapping("/{ID}")
    public R<InoutCheckVo> getInfo(@NotNull(message = "主键不能为空")
                                   @PathVariable String ID) {
        return R.ok(inoutCheckService.queryById(ID));
    }

    /**
     * 新增收支对账
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    @Log(title = "收支对账", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody InoutCheckBo bo) {
        return toAjax(inoutCheckService.insertByBo(bo));
    }

    /**
     * 修改收支对账
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    @Log(title = "收支对账", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody InoutCheckBo bo) {
        return toAjax(inoutCheckService.updateByBo(bo));
    }

    /**
     * 删除收支对账
     *
     * @param IDs 主键串
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    @Log(title = "收支对账", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{IDs}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable String[] IDs) {
        return toAjax(inoutCheckService.deleteWithValidByIds(List.of(IDs), true));
    }

    /**
     * 财务审计支持-列表
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    @GetMapping("/productList")
    public TableDataInfo<ProductPanoramaVo> productPanorama(ProductPanoramaVo vo, PageQuery pageQuery) {
        return zjkProductService.productPanorama(vo, pageQuery);
    }

    /**
     * 财务审计支持-查看
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    @GetMapping("/checkInfoList")
    public TableDataInfo<CheckSupportVo> checkInfoList(CheckSupportVo bo, PageQuery pageQuery) {
        return inoutCheckService.checkInfoList(bo, pageQuery);
    }

    /**
     * 财务审计支持-查看导出
     */
    @DecryptOperation
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    @PostMapping("/export")
    public void export(CheckSupportVo bo, HttpServletResponse response) {
        List<CheckSupportVo> list = inoutCheckService.checkInfoListToExpert(bo);
        ExcelUtil.exportExcel(list, "导出", CheckSupportVo.class, response);
    }
    /**
     * 获取发票id
     *
     * @param bo
     */
    @Log(title = "获取发票id", businessType = BusinessType.EXPORT)
    @GetMapping("/getFpIds")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    public  R<List<String>> downloadToZip(CheckSupportVo bo) {
        List<String> strings = inoutCheckService.downloadFpToZip(bo);
        return  R.ok( strings);
    }
}
