package org.dromara.zjk.statement.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.statement.domain.vo.ZjkMoneyAnalyseStatementVo;
import org.dromara.zjk.statement.domain.bo.ZjkMoneyAnalyseStatementBo;
import org.dromara.zjk.statement.service.IZjkMoneyAnalyseStatementService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 报表分析
 * 前端访问路由地址为:/zjk/moneyAnalyseStatement
 *
 * @author ygh
 * @date 2024-12-16
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/moneyAnalyseStatement")
public class ZjkMoneyAnalyseStatementController extends BaseController {

    private final IZjkMoneyAnalyseStatementService zjkMoneyAnalyseStatementService;

    /**
     * 查询报表分析列表
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkMoneyAnalyseStatementVo> list(ZjkMoneyAnalyseStatementBo bo, PageQuery pageQuery) {
        return zjkMoneyAnalyseStatementService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出报表分析列表
     */
    @SaCheckPermission("zjk:moneyAnalyseStatement:export")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "报表分析", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkMoneyAnalyseStatementBo bo, HttpServletResponse response) {
        List<ZjkMoneyAnalyseStatementVo> list = zjkMoneyAnalyseStatementService.queryList(bo);
        ExcelUtil.exportExcel(list, "报表分析", ZjkMoneyAnalyseStatementVo.class, response);
    }

    /**
     * 获取报表分析详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("zjk:moneyAnalyseStatement:query")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkMoneyAnalyseStatementVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkMoneyAnalyseStatementService.queryById(id));
    }

    /**
     * 新增报表分析
     */
    @SaCheckPermission("zjk:moneyAnalyseStatement:add")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "报表分析", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkMoneyAnalyseStatementBo bo) {
        return toAjax(zjkMoneyAnalyseStatementService.insertByBo(bo));
    }

    /**
     * 修改报表分析
     */
    @SaCheckPermission("zjk:moneyAnalyseStatement:edit")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "报表分析", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkMoneyAnalyseStatementBo bo) {
        return toAjax(zjkMoneyAnalyseStatementService.updateByBo(bo));
    }

    /**
     * 删除报表分析
     *
     * @param ids 主键串
     */
    @SaCheckPermission("zjk:moneyAnalyseStatement:remove")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "报表分析", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkMoneyAnalyseStatementService.deleteWithValidByIds(List.of(ids), true));
    }
}
