package org.dromara.zjk.utils;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;

public class IdCardValidator {

    /**
     * 校验身份证号是否合法（支持18位）
     */
    public static boolean isValid(String idCard) {
        if (idCard == null || idCard.length() != 18) {
            return false;
        }

        // 前17位必须是数字
        String body = idCard.substring(0, 17);
        if (!body.matches("\\d+")) {
            return false;
        }

        // 校验出生日期是否合法
        String birth = idCard.substring(6, 14); // yyyyMMdd
        if (!isValidDate(birth)) {
            return false;
        }

        // 校验最后一位校验码
        return verifyCheckDigit(idCard);
    }

    private static boolean isValidDate(String dateStr) {
        try {
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");
            LocalDate date = LocalDate.parse(dateStr, formatter);

            // 出生日期不能超过当前日期，不能太早（如小于1900）
            return !date.isAfter(LocalDate.now()) && date.getYear() >= 1900;
        } catch (DateTimeParseException e) {
            return false;
        }
    }

    private static boolean verifyCheckDigit(String idCard) {
        // 系数数组
        int[] factor = {7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2};
        // 校验位数组
        char[] checkCode = {'1', '0', 'X', '9', '8', '7', '6', '5', '4', '3', '2'};

        int sum = 0;
        for (int i = 0; i < 17; i++) {
            sum += (idCard.charAt(i) - '0') * factor[i];
        }
        int mod = sum % 11;
        return idCard.charAt(17) == checkCode[mod];
    }
}

