package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkCreditBo;
import org.dromara.zjk.domain.bo.ZjkCreditUserBo;
import org.dromara.zjk.domain.vo.ChartBaseCreditVO;
import org.dromara.zjk.domain.vo.CreditExpertRankVO;
import org.dromara.zjk.domain.vo.ZjkCreditRuleVo;
import org.dromara.zjk.domain.vo.ZjkCreditUserVo;
import org.dromara.zjk.service.IZjkCreditUserService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 信用用户管理
 * 前端访问路由地址为:/zjk/creditUser
 *
 * @author ygh
 * @date 2024-11-11
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/creditUser")
public class ZjkCreditUserController extends BaseController {

    private final IZjkCreditUserService zjkCreditUserService;

    /**
     * 查询信用用户管理列表
     */
//    @SaCheckPermission("zjk:creditUser:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkCreditUserVo> list(ZjkCreditUserBo bo, PageQuery pageQuery) {
        return zjkCreditUserService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出信用用户管理列表
     */
    @SaCheckPermission("zjk:creditUser:export")
    @Log(title = "信用用户管理", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkCreditUserBo bo, HttpServletResponse response) {
        List<ZjkCreditUserVo> list = zjkCreditUserService.queryList(bo);
        ExcelUtil.exportExcel(list, "信用用户管理", ZjkCreditUserVo.class, response);
    }

    /**
     * 获取信用用户管理详细信息
     *
     * @param ID 主键
     */
    @SaCheckPermission("zjk:creditUser:query")
    @GetMapping("/{ID}")
    public R<ZjkCreditUserVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable String ID) {
        return R.ok(zjkCreditUserService.queryById(ID));
    }

    /**
     * 新增信用用户管理
     */
    @SaCheckPermission("zjk:creditUser:add")
    @Log(title = "信用用户管理", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkCreditUserBo bo) {
        return toAjax(zjkCreditUserService.insertByBo(bo));
    }

    /**
     * 修改信用用户管理
     */
    @SaCheckPermission("zjk:creditUser:edit")
    @Log(title = "信用用户管理", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkCreditUserBo bo) {
        return toAjax(zjkCreditUserService.updateByBo(bo));
    }

    /**
     * 删除信用用户管理
     *
     * @param IDs 主键串
     */
    @SaCheckPermission("zjk:creditUser:remove")
    @Log(title = "信用用户管理", businessType = BusinessType.DELETE)
    @DeleteMapping("/{IDs}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable String[] IDs) {
        return toAjax(zjkCreditUserService.deleteWithValidByIds(List.of(IDs), true));
    }

    /**
     * 专家积分榜基本信息
     */
    @GetMapping("/expertPointRank")
    public  R<CreditExpertRankVO> expertPointRank(Integer rangeDay) {
        return zjkCreditUserService.getExpertPointRank(rangeDay);
    }

    //专家信用详情
    @GetMapping("/selectUserById")
    public  R<List<ZjkCreditRuleVo>> expertPointRanks(ZjkCreditBo creditId, PageQuery pageQuery) {
        return zjkCreditUserService.selectUserById(creditId,pageQuery);
    }

    //黑名单
    @GetMapping("/blacklist")
    public  R<List<ChartBaseCreditVO>> blacklist(ZjkCreditBo creditBo) {
        return zjkCreditUserService.blacklist(creditBo);
    }

    /*出库*/
    @PostMapping("/outCreditId")
    public  R<Void> outCreditId(@RequestBody ZjkCreditUserBo bo) {
        return toAjax(zjkCreditUserService.creditId(bo));
    }

    /**
     * 手动移入黑名单
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    @Log(title = "手动移入黑名单", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/moveToBlacklist")
    public R<Void> moveToBlacklist(@RequestBody ZjkCreditUserBo bo) {
        return toAjax(zjkCreditUserService.moveToBlacklist(bo));
    }


    /**
     * 手动移出黑名单
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    }, mode = SaMode.OR)
    @Log(title = "手动移出黑名单", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/removeTheBlacklist")
    public R<Void> removeTheBlacklist(@RequestBody ZjkCreditUserBo bo) {
        return toAjax(zjkCreditUserService.removeTheBlacklist(bo));
    }




    /**
     * 获取黑名单专家总人数
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/getBlacklistCount")
    public R<Long> getBlacklistCount() {
        return R.ok("success",zjkCreditUserService.getBlacklistCount());
    }


    //黑名单
    @SaCheckRole(value = {
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/getCount")
    public  R<Long> getCount() {
        return R.ok("success",zjkCreditUserService.getCount());
    }
}
