package org.dromara.zjk.expert.expertPatent.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.expert.expertPatent.domain.bo.ZjkExpertPatentBo;
import org.dromara.zjk.expert.expertPatent.domain.vo.ZjkExpertPatentVo;
import org.dromara.zjk.expert.expertPatent.service.IZjkExpertPatentService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 专家专利信息
 * 前端访问路由地址为:/expert/expertPatent
 *
 * @author zhangzhou
 * @date 2024-11-10
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/expertPatent")
public class ZjkExpertPatentController extends BaseController {

    private final IZjkExpertPatentService zjkExpertPatentService;

    /**
     * 查询专家专利信息列表
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkExpertPatentVo> list(ZjkExpertPatentBo bo, PageQuery pageQuery) {
        return zjkExpertPatentService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出专家专利信息列表
     */
    @SaCheckPermission("expert:expertPatent:export")
    @Log(title = "专家专利信息", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkExpertPatentBo bo, HttpServletResponse response) {
        List<ZjkExpertPatentVo> list = zjkExpertPatentService.queryList(bo);
        ExcelUtil.exportExcel(list, "专家专利信息", ZjkExpertPatentVo.class, response);
    }

    /**
     * 获取专家专利信息详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("expert:expertPatent:query")
    @GetMapping("/{id}")
    public R<ZjkExpertPatentVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkExpertPatentService.queryById(id));
    }

    /**
     * 新增专家专利信息
     */
    @SaCheckPermission("expert:expertPatent:add")
    @Log(title = "专家专利信息", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkExpertPatentBo bo) {
        return toAjax(zjkExpertPatentService.insertByBo(bo));
    }

    /**
     * 修改专家专利信息
     */
    @SaCheckPermission("expert:expertPatent:edit")
    @Log(title = "专家专利信息", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkExpertPatentBo bo) {
        return toAjax(zjkExpertPatentService.updateByBo(bo));
    }

    /**
     * 删除专家专利信息
     *
     * @param ids 主键串
     */
    @SaCheckPermission("expert:expertPatent:remove")
    @Log(title = "专家专利信息", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkExpertPatentService.deleteWithValidByIds(List.of(ids), true));
    }

    /**
     * 通过项目id  喝项目阶段 获取
     * @param productId
     * @param reviewSn
     * @return
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/patent/list")
    public R<List<ZjkExpertPatentVo>> paperList(Long productId, String reviewSn) {
        return R.ok(zjkExpertPatentService.patentList(productId, reviewSn));
    }


    /**
     * 获取在库专利总数
     * @return
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("getExpertPatentCount")
    public R<Long> getExpertPatentCount(){
        return R.ok(zjkExpertPatentService.getExpertPatentCount());
    }
}
