package org.dromara.zjk.statement.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.statement.domain.vo.ZjkMoneyExpertStatementVo;
import org.dromara.zjk.statement.domain.bo.ZjkMoneyExpertStatementBo;
import org.dromara.zjk.statement.service.IZjkMoneyExpertStatementService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 专家费用报表
 * 前端访问路由地址为:/zjk/moneyExpertStatement
 *
 * @author ygh
 * @date 2024-12-16
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/moneyExpertStatement")
public class ZjkMoneyExpertStatementController extends BaseController {

    private final IZjkMoneyExpertStatementService zjkMoneyExpertStatementService;

    /**
     * 查询专家费用报表列表
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkMoneyExpertStatementVo> list(ZjkMoneyExpertStatementBo bo, PageQuery pageQuery) {
        return zjkMoneyExpertStatementService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出专家费用报表列表
     */
    @SaCheckPermission("zjk:moneyExpertStatement:export")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "专家费用报表", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkMoneyExpertStatementBo bo, HttpServletResponse response) {
        List<ZjkMoneyExpertStatementVo> list = zjkMoneyExpertStatementService.queryList(bo);
        ExcelUtil.exportExcel(list, "专家费用报表", ZjkMoneyExpertStatementVo.class, response);
    }

    /**
     * 获取专家费用报表详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("zjk:moneyExpertStatement:query")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkMoneyExpertStatementVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkMoneyExpertStatementService.queryById(id));
    }

    /**
     * 新增专家费用报表
     */
    @SaCheckPermission("zjk:moneyExpertStatement:add")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "专家费用报表", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkMoneyExpertStatementBo bo) {
        return toAjax(zjkMoneyExpertStatementService.insertByBo(bo));
    }

    /**
     * 修改专家费用报表
     */
    @SaCheckPermission("zjk:moneyExpertStatement:edit")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "专家费用报表", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkMoneyExpertStatementBo bo) {
        return toAjax(zjkMoneyExpertStatementService.updateByBo(bo));
    }

    /**
     * 删除专家费用报表
     *
     * @param ids 主键串
     */
    @SaCheckPermission("zjk:moneyExpertStatement:remove")
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "专家费用报表", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkMoneyExpertStatementService.deleteWithValidByIds(List.of(ids), true));
    }
}
