package org.dromara.zjk.controller;

import java.util.List;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import lombok.RequiredArgsConstructor;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.*;
import cn.dev33.satoken.annotation.SaCheckPermission;
import org.dromara.common.core.constant.TenantConstants;
import org.springframework.web.bind.annotation.*;
import org.springframework.validation.annotation.Validated;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.web.core.BaseController;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.zjk.domain.vo.LearningBackVo;
import org.dromara.zjk.domain.bo.LearningBackBo;
import org.dromara.zjk.service.ILearningBackService;
import org.dromara.common.mybatis.core.page.TableDataInfo;

/**
 * 学习园地-学习反馈
 * 前端访问路由地址为:/zjk/learningBack
 *
 * @author ygh
 * @date 2025-07-28
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningBack")
public class LearningBackController extends BaseController {

    private final ILearningBackService learningBackService;

    /**
     * 查询学习园地-学习反馈列表
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<LearningBackVo> list(LearningBackBo bo, PageQuery pageQuery) {
        return learningBackService.queryPageList(bo, pageQuery);
    }

    /**
     * 学习园地-学习反馈历史记录列表
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习反馈", businessType = BusinessType.VIEW)
    @GetMapping("/myList")
    public R< List<LearningBackVo>> export(LearningBackBo bo, HttpServletResponse response) {
        List<LearningBackVo> list = learningBackService.queryList(bo);
      return R.ok(list);
    }

    /**
     * 获取学习园地-学习反馈详细信息
     *
     * @param ID 主键
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @GetMapping("/{ID}")
    public R<LearningBackVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable String ID) {
        return R.ok(learningBackService.queryById(ID));
    }

    /**
     * 新增学习园地-学习反馈
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习反馈", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody LearningBackBo bo) {
        return toAjax(learningBackService.insertByBo(bo));
    }

    /**
     * 修改学习园地-学习反馈
     */
    @SaCheckRole(value = {
        TenantConstants.EXPERT_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习反馈", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody LearningBackBo bo) {
        return toAjax(learningBackService.updateByBo(bo));
    }

    /**
     * 删除学习园地-学习反馈
     *
     * @param IDs 主键串
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    },mode = SaMode.OR)
    @Log(title = "学习园地-学习反馈", businessType = BusinessType.DELETE)
    @GetMapping("/delete/{IDs}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable String[] IDs) {
        return toAjax(learningBackService.deleteWithValidByIds(List.of(IDs), true));
    }
}
