package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import cn.hutool.core.bean.BeanUtil;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.ZjkExpert;
import org.dromara.zjk.domain.bo.ZjkProductSummarizeBo;
import org.dromara.zjk.domain.excels.ZjkProductSummarizeExcelVO;
import org.dromara.zjk.domain.vo.ZjkProductSummarizeExportVo;
import org.dromara.zjk.domain.vo.ZjkProductSummarizeVo;
import org.dromara.zjk.service.IZjkProductSummarizeService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * 项目总结
 * 前端访问路由地址为:/zjk/productSummarize
 *
 * @author ygh
 * @date 2024-11-02
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/productSummarize")
public class ZjkProductSummarizeController extends BaseController {

    private final IZjkProductSummarizeService zjkProductSummarizeService;

    /**
     * 查询项目总结列表
     */
    @SaCheckPermission("zjk:productSummarize:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkProductSummarizeVo> list(ZjkProductSummarizeBo bo, PageQuery pageQuery) {
        return zjkProductSummarizeService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出项目总结列表
     */
    @Log(title = "项目总结", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    @SaCheckPermission("zjk:productSummarize:export")
    public void export(ZjkProductSummarizeBo bo, HttpServletResponse response) {
        List<ZjkProductSummarizeVo> list = zjkProductSummarizeService.queryList(bo);
        List<ZjkProductSummarizeExportVo> zjkProductSummarizeExportVos = BeanUtil.copyToList(list, ZjkProductSummarizeExportVo.class);

        ExcelUtil.exportExcel(zjkProductSummarizeExportVos, "项目总结", ZjkProductSummarizeExportVo.class, response);
    }
    /**
     * 导出项目总结列表
     */
    @Log(title = "项目总结模板", businessType = BusinessType.EXPORT)
    @PostMapping("/export/template")
    @SaCheckRole(value =
        {
            TenantConstants.XM_ROLE_KEY,
            TenantConstants.TENANT_ADMIN_ROLE_KEY,
            TenantConstants.JG_ROLE_KEY,
        }
        , mode = SaMode.OR)
    public void exportTemplate(HttpServletResponse response) {
        ExcelUtil.exportExcel(new ArrayList<>(), "项目总结分类", ZjkProductSummarizeExcelVO.class, response);
    }

    /**
     * 导出项目总结列表
     */
    @Log(title = "项目总结模板", businessType = BusinessType.EXPORT)
    @PostMapping("/export/import")
    @SaCheckRole(value =
        {
            TenantConstants.XM_ROLE_KEY,
            TenantConstants.TENANT_ADMIN_ROLE_KEY,
            TenantConstants.JG_ROLE_KEY,
        }
        , mode = SaMode.OR)
    public R exportImport(@RequestPart("file") MultipartFile file) throws IOException {

        InputStream inputStream = file.getInputStream();
        // 文件转化
        List<ZjkProductSummarizeExcelVO> vos = ExcelUtil.importExcel(inputStream, ZjkProductSummarizeExcelVO.class);


        boolean b = zjkProductSummarizeService.importSummarize(vos);

        return toAjax(b);
    }

    /**
     * 获取项目总结详细信息
     *
     * @param id 主键
     */
    @DecryptOperation
    @GetMapping("/{id}")
    @SaCheckRole(value =
        {
            TenantConstants.XM_ROLE_KEY,
            TenantConstants.TENANT_ADMIN_ROLE_KEY,
            TenantConstants.JG_ROLE_KEY,
        }
        , mode = SaMode.OR)
    public R<ZjkProductSummarizeVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkProductSummarizeService.queryById(id));
    }


    @DecryptOperation
    @GetMapping("/get-expert-by-review-id")
    public R<List<ZjkExpert>> getExpertByReviewId(Long reviewId) {
        return R.ok("",zjkProductSummarizeService.getExpertByReviewId(reviewId));
    }
    /**
     * 新增项目总结
     */
    @RepeatSubmit()
    @SaCheckPermission("zjk:productSummarize:add")
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkProductSummarizeBo bo) {
        return toAjax(zjkProductSummarizeService.insertByBo(bo));
    }

    /**
     * 修改项目总结
     */
    @Log(title = "项目总结", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update")
    @SaCheckPermission("zjk:productSummarize:update")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkProductSummarizeBo bo) {
        return toAjax(zjkProductSummarizeService.updateByBo(bo));
    }

    /**
     * 删除项目总结
     *
     * @param ids 主键串
     */
    @Log(title = "项目总结", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkProductSummarizeService.deleteWithValidByIds(List.of(ids), true));
    }

//    /**
//     * 项目总结导入
//     * @param file
//     * @return
//     */
//    @SaCheckPermission("expert:expert:import")
//    @Log(title = "专家库-专家信息导入", businessType = BusinessType.IMPORT)
//    @PostMapping("/imports")
//    public R<Void> upload(@RequestPart("file") MultipartFile file) {
//        try {
//            InputStream inputStream = file.getInputStream();
//            // 文件转化
//            List<ZjkProductSummarizeBo> zjkExpertVos = ExcelUtil.importExcel(inputStream, ZjkProductSummarizeBo.class);
//            // 提取所有的身份证号
//            List<String> idCards = zjkExpertVos.stream()
//                .map(ZjkProductSummarizeBo::getIdCard)
//                .collect(Collectors.toList());
//            // 根据提取的 idCards 列表查询数据库中已存在的 idCards
//            List<String> existingIdCards = zjkExpertService.selectExistingIdCards(idCards);
//            // 如果存在重复的身份证号则提示，剔除之后再次导入
//            if (!CollUtil.isEmpty(existingIdCards)){
//                // 将身份证号列表转换为用逗号分隔的字符串
//                return R.fail("以下身份证号专家已经存在，请剔除后再次导入专家数据：" + String.join(", ", existingIdCards));
//            }
//            // 进行数据新增
//            if (!zjkExpertService.insertList(zjkExpertVos)) {
//                return R.fail();
//            }
//        }catch (Exception e) {
//            return R.fail("导入失败请联系管理员");
//        }
//        return R.ok();
//    }

    /*
    项目总结评价
    * */
    /**
     * 修改项目总结
     */
    @SaCheckPermission("zjk:productSummarize:projectEvaluation")
    @Log(title = "项目总结评价", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/projectEvaluation")
    public R<Void> projectEvaluation(@Validated(EditGroup.class) @RequestBody ZjkProductSummarizeBo bo) {
        return toAjax(zjkProductSummarizeService.updateByBo(bo));
    }

    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/getCount")
    public R<Long> getProductSummarizeCount() {
        return R.ok(zjkProductSummarizeService.getCount());
    }
}
