package org.dromara.zjk.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.exception.ServiceException;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.ServletUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.zjk.domain.ZjkDataChangesLogs;
import org.dromara.zjk.domain.bo.ZjkDataChangesLogsBo;
import org.dromara.zjk.domain.vo.ZjkDataChangesLogsDetailVo;
import org.dromara.zjk.domain.vo.ZjkDataChangesLogsVo;
import org.dromara.zjk.mapper.ZjkDataChangesLogsMapper;
import org.dromara.zjk.service.IZjkDataChangesLogsService;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 业务数据变更记录Service业务层处理
 *
 * @author ygh
 * @date 2024-11-07
 */
@RequiredArgsConstructor
@Service
public class ZjkDataChangesLogsServiceImpl implements IZjkDataChangesLogsService {

    private final ZjkDataChangesLogsMapper baseMapper;

    /**
     * 查询业务数据变更记录
     *
     * @param id 主键
     * @return 业务数据变更记录
     */
    @Override
    public ZjkDataChangesLogsDetailVo queryById(Long id){
        ZjkDataChangesLogsVo zjkDataChangesLogsVo = baseMapper.selectVoById(id);
        if (zjkDataChangesLogsVo == null) {
            throw new ServiceException("数据异常");
        }
        ZjkDataChangesLogsDetailVo zjkDataChangesLogsDetailVo = new ZjkDataChangesLogsDetailVo();
        zjkDataChangesLogsDetailVo.setId(zjkDataChangesLogsVo.getId());
        zjkDataChangesLogsDetailVo.setBusinessType(zjkDataChangesLogsVo.getBusinessType());
        zjkDataChangesLogsDetailVo.setChangeType(zjkDataChangesLogsVo.getChangeType());
        zjkDataChangesLogsDetailVo.setOperIp(zjkDataChangesLogsVo.getOperIp());
        zjkDataChangesLogsDetailVo.setCreateBy(zjkDataChangesLogsVo.getCreateBy());
        zjkDataChangesLogsDetailVo.setCreateTime(zjkDataChangesLogsVo.getCreateTime());
        if (StringUtils.isNotBlank(zjkDataChangesLogsVo.getBeforeOper())) {
            zjkDataChangesLogsDetailVo.setBeforeChange(zjkDataChangesLogsVo.getBeforeOper());
        }
        if (StringUtils.isNotBlank(zjkDataChangesLogsVo.getAfterOper())) {
            zjkDataChangesLogsDetailVo.setAfterChange(zjkDataChangesLogsVo.getAfterOper());
        }
        return zjkDataChangesLogsDetailVo;
    }

    /**
     * 分页查询业务数据变更记录列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 业务数据变更记录分页列表
     */
    @Override
    public TableDataInfo<ZjkDataChangesLogsVo> queryPageList(ZjkDataChangesLogsBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkDataChangesLogs> lqw = buildQueryWrapper(bo);
        Page<ZjkDataChangesLogsVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的业务数据变更记录列表
     *
     * @param bo 查询条件
     * @return 业务数据变更记录列表
     */
    @Override
    public List<ZjkDataChangesLogsVo> queryList(ZjkDataChangesLogsBo bo) {
        LambdaQueryWrapper<ZjkDataChangesLogs> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkDataChangesLogs> buildQueryWrapper(ZjkDataChangesLogsBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkDataChangesLogs> lqw = Wrappers.lambdaQuery();
        lqw.eq(StringUtils.isNotBlank(bo.getChangeType()), ZjkDataChangesLogs::getChangeType, bo.getChangeType());
        lqw.eq(StringUtils.isNotBlank(bo.getOperIp()), ZjkDataChangesLogs::getOperIp, bo.getOperIp());
        lqw.between(bo.getStartTime()!=null && bo.getEndTime() != null,
            ZjkDataChangesLogs::getCreateTime ,bo.getStartTime(), bo.getEndTime());
        return lqw;
    }

    /**
     * 新增业务数据变更记录
     *
     * @param bo 业务数据变更记录
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkDataChangesLogsBo bo) {
        ZjkDataChangesLogs add = MapstructUtils.convert(bo, ZjkDataChangesLogs.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改业务数据变更记录
     *
     * @param bo 业务数据变更记录
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkDataChangesLogsBo bo) {
        ZjkDataChangesLogs update = MapstructUtils.convert(bo, ZjkDataChangesLogs.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkDataChangesLogs entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除业务数据变更记录信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public Boolean insertDataChangesLogs(ZjkDataChangesLogs zjkDataChangesLogs) {
        zjkDataChangesLogs.setOperIp(ServletUtils.getClientIP());
        return baseMapper.insert(zjkDataChangesLogs)>0;
    }
}
