package org.dromara.zjk.expert.expertAwards.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.zjk.expert.expertAwards.domain.ZjkExpertAwards;
import org.dromara.zjk.expert.expertAwards.domain.bo.ZjkExpertAwardsBo;
import org.dromara.zjk.expert.expertAwards.domain.vo.ZjkExpertAwardsVo;
import org.dromara.zjk.expert.expertAwards.mapper.ZjkExpertAwardsMapper;
import org.dromara.zjk.expert.expertAwards.service.IZjkExpertAwardsService;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 专家获奖情况Service业务层处理
 *
 * @author zhangzhou
 * @date 2024-11-10
 */
@RequiredArgsConstructor
@Service
public class ZjkExpertAwardsServiceImpl implements IZjkExpertAwardsService {

    private final ZjkExpertAwardsMapper baseMapper;

    /**
     * 查询专家获奖情况
     *
     * @param id 主键
     * @return 专家获奖情况
     */
    @Override
    public ZjkExpertAwardsVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询专家获奖情况列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 专家获奖情况分页列表
     */
    @Override
    public TableDataInfo<ZjkExpertAwardsVo> queryPageList(ZjkExpertAwardsBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkExpertAwards> lqw = buildQueryWrapper(bo);
        Page<ZjkExpertAwardsVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的专家获奖情况列表
     *
     * @param bo 查询条件
     * @return 专家获奖情况列表
     */
    @Override
    public List<ZjkExpertAwardsVo> queryList(ZjkExpertAwardsBo bo) {
        LambdaQueryWrapper<ZjkExpertAwards> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkExpertAwards> buildQueryWrapper(ZjkExpertAwardsBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkExpertAwards> lqw = Wrappers.lambdaQuery();
        lqw.eq(bo.getExpertId() != null, ZjkExpertAwards::getExpertId, bo.getExpertId());
        lqw.eq(StringUtils.isNotBlank(bo.getAwardsPath()), ZjkExpertAwards::getAwardsPath, bo.getAwardsPath());
        lqw.eq(StringUtils.isNotBlank(bo.getAwardsContent()), ZjkExpertAwards::getAwardsContent, bo.getAwardsContent());
        return lqw;
    }

    /**
     * 新增专家获奖情况
     *
     * @param bo 专家获奖情况
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkExpertAwardsBo bo) {
        ZjkExpertAwards add = MapstructUtils.convert(bo, ZjkExpertAwards.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改专家获奖情况
     *
     * @param bo 专家获奖情况
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkExpertAwardsBo bo) {
        ZjkExpertAwards update = MapstructUtils.convert(bo, ZjkExpertAwards.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkExpertAwards entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除专家获奖情况信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public void handleExpertAwards(List<ZjkExpertAwards> zjkExpertAwardsList, Long expertId) {
        baseMapper.delete(new LambdaQueryWrapper<ZjkExpertAwards>()
            .eq(ZjkExpertAwards::getExpertId, expertId)
        );
        if (zjkExpertAwardsList != null && !zjkExpertAwardsList.isEmpty()){
            zjkExpertAwardsList.forEach(zjkExpertAwards -> {
                zjkExpertAwards.setExpertId(expertId);
            });
            baseMapper.insertOrUpdateBatch(zjkExpertAwardsList);
        }


    }

    @Override
    public List<ZjkExpertAwards> populateExpertAwardsList(Long expertId) {
        List<ZjkExpertAwards> zjkExpertAwardsList = baseMapper.selectList(
            new LambdaQueryWrapper<ZjkExpertAwards>()
                .eq(ZjkExpertAwards::getExpertId, expertId)
        );
        return zjkExpertAwardsList.isEmpty() ? new ArrayList<>() : zjkExpertAwardsList;
    }

    @Override
    public List<ZjkExpertAwards> selectByExpertId(Long expertId) {
        return baseMapper.selectByExpertId(expertId);
    }
}
