package org.dromara.zjk.learning_world.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import org.dromara.common.core.exception.ServiceException;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.mybatis.core.page.PageQuery;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import lombok.RequiredArgsConstructor;
import org.dromara.common.satoken.utils.LoginHelper;
import org.dromara.zjk.learning_world.domain.excel.ZjkLearningResourceExcel;
import org.dromara.zjk.enums.PublishStatusEnum;
import org.dromara.zjk.enums.SourceClassifyEnum;
import org.dromara.zjk.enums.SourceTypeEnum;
import org.dromara.zjk.utils.EnumUtil;
import org.springframework.stereotype.Service;
import org.dromara.zjk.learning_world.domain.bo.ZjkLearningResourceBo;
import org.dromara.zjk.learning_world.domain.vo.ZjkLearningResourceVo;
import org.dromara.zjk.learning_world.domain.entity.ZjkLearningResource;
import org.dromara.zjk.learning_world.mapper.ZjkLearningResourceMapper;
import org.dromara.zjk.learning_world.service.IZjkLearningResourceService;

import java.util.*;

/**
 * 资源管理Service业务层处理
 *
 * @author ygh
 * @date 2024-11-08
 */
@RequiredArgsConstructor
@Service
public class ZjkLearningResourceServiceImpl implements IZjkLearningResourceService {

    private final ZjkLearningResourceMapper baseMapper;

    /**
     * 查询资源管理
     *
     * @param id 主键
     * @return 资源管理
     */
    @Override
    public ZjkLearningResourceVo queryById(Long id) {
        return baseMapper.getById(id);
    }

    /**
     * 分页查询资源管理列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 资源管理分页列表
     */
    @Override
    public TableDataInfo<ZjkLearningResourceVo> queryPageList(ZjkLearningResourceBo bo, PageQuery pageQuery) {
        LambdaQueryWrapper<ZjkLearningResource> lqw = buildQueryWrapper(bo);
        Page<ZjkLearningResourceVo> result = baseMapper.selectVoPage(pageQuery.build(), lqw);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的资源管理列表
     *
     * @param bo 查询条件
     * @return 资源管理列表
     */
    @Override
    public List<ZjkLearningResourceVo> queryList(ZjkLearningResourceBo bo) {
        LambdaQueryWrapper<ZjkLearningResource> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkLearningResource> buildQueryWrapper(ZjkLearningResourceBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkLearningResource> lqw = Wrappers.lambdaQuery();
        lqw.like(StringUtils.isNotBlank(bo.getResourceName()), ZjkLearningResource::getResourceName, bo.getResourceName());
        lqw.eq(StringUtils.isNotBlank(bo.getResourceClassify()), ZjkLearningResource::getResourceClassify, bo.getResourceClassify());
        lqw.eq(StringUtils.isNotBlank(bo.getResourceType()), ZjkLearningResource::getResourceType, bo.getResourceType());
        lqw.eq(StringUtils.isNotBlank(bo.getResourceBrief()), ZjkLearningResource::getResourceBrief, bo.getResourceBrief());
        lqw.eq(bo.getResourceTime() != null, ZjkLearningResource::getResourceTime, bo.getResourceTime());
        lqw.eq(StringUtils.isNotBlank(bo.getAuthor()), ZjkLearningResource::getAuthor, bo.getAuthor());
        lqw.eq(StringUtils.isNotBlank(bo.getPublishJournal()), ZjkLearningResource::getPublishJournal, bo.getPublishJournal());
        lqw.eq(StringUtils.isNotBlank(bo.getJournalNo()), ZjkLearningResource::getJournalNo, bo.getJournalNo());
        lqw.eq(StringUtils.isNotBlank(bo.getPaperTitle()), ZjkLearningResource::getPaperTitle, bo.getPaperTitle());
        lqw.eq(StringUtils.isNotBlank(bo.getPaperAccessory()), ZjkLearningResource::getPaperAccessory, bo.getPaperAccessory());
        lqw.eq(StringUtils.isNotBlank(bo.getStatus()), ZjkLearningResource::getStatus, bo.getStatus());
        lqw.ge(!ObjectUtil.isEmpty(bo.getStartTime()), ZjkLearningResource::getResourceTime, bo.getStartTime());
        lqw.le(!ObjectUtil.isEmpty(bo.getEndTime()), ZjkLearningResource::getResourceTime, bo.getEndTime());
        return lqw;
    }

    /**
     * 新增资源管理
     *
     * @param bo 资源管理
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkLearningResourceBo bo) {
        ZjkLearningResource add = MapstructUtils.convert(bo, ZjkLearningResource.class);
        add.setUserId(LoginHelper.getUserId());
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改资源管理
     *
     * @param bo 资源管理
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkLearningResourceBo bo) {
        ZjkLearningResource update = MapstructUtils.convert(bo, ZjkLearningResource.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkLearningResource entity) {
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除资源管理信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if (isValid) {
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public List<ZjkLearningResourceExcel> exportList(ZjkLearningResourceBo bo) {
        List<ZjkLearningResourceVo> zjkLearningResourceVos = queryList(bo);
        if (CollectionUtil.isEmpty(zjkLearningResourceVos)) {
            return new ArrayList<>();
        }
        List<ZjkLearningResourceExcel> zjkLearningResourceExcels = BeanUtil.copyToList(zjkLearningResourceVos, ZjkLearningResourceExcel.class);
        for (ZjkLearningResourceExcel zjkLearningResourceExcel : zjkLearningResourceExcels) {
            String resourceType = zjkLearningResourceExcel.getResourceType();
            if (StringUtils.isNotBlank(resourceType)) {
                String nameByCode = EnumUtil.getNameByCode(SourceTypeEnum.class, resourceType);
                zjkLearningResourceExcel.setResourceTypeName(nameByCode);
            }
            String status = zjkLearningResourceExcel.getStatus();
            if (StringUtils.isNotBlank(status)) {
                String statusByCode = EnumUtil.getNameByCode(PublishStatusEnum.class, status);
                zjkLearningResourceExcel.setStatusName(statusByCode);
            }
            String resourceClassify = zjkLearningResourceExcel.getResourceClassify();
            if (StringUtils.isNotBlank(resourceClassify)) {
                String resourceClassifyByCode = EnumUtil.getNameByCode(SourceClassifyEnum.class, resourceClassify);
                zjkLearningResourceExcel.setResourceClassifyName(resourceClassifyByCode);
            }
        }

        return List.of();
    }

    @Override
    public int updateStatus(ZjkLearningResourceBo bo) {
        Long id = bo.getId();
        ZjkLearningResource byId = baseMapper.selectById(id);
        if (byId == null) {
            throw new ServiceException("无效id");
        }
        byId.setStatus(bo.getStatus());
        byId.setTakedownTime(new Date());
        return baseMapper.updateById(byId);
    }
}
