package org.dromara.common.excel.convert;

import cn.hutool.core.annotation.AnnotationUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.excel.converters.Converter;
import com.alibaba.excel.enums.CellDataTypeEnum;
import com.alibaba.excel.metadata.GlobalConfiguration;
import com.alibaba.excel.metadata.data.ReadCellData;
import com.alibaba.excel.metadata.data.WriteCellData;
import com.alibaba.excel.metadata.property.ExcelContentProperty;
import org.dromara.common.excel.annotation.ExcelDictFormat;
import org.dromara.common.core.service.DictService;
import org.dromara.common.core.utils.SpringUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.excel.utils.ExcelUtil;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

/**
 * 多选字典格式化转换处理
 *
 * @author your-name
 */
@Slf4j
public class MultiDictConvert implements Converter<Object> {

    @Override
    public Class<Object> supportJavaTypeKey() {
        return Object.class;
    }

    @Override
    public CellDataTypeEnum supportExcelTypeKey() {
        return null;
    }

    @Override
    public Object convertToJavaData(ReadCellData<?> cellData, ExcelContentProperty contentProperty, GlobalConfiguration globalConfiguration) {
        ExcelDictFormat anno = getAnnotation(contentProperty.getField());
        String type = anno.dictType();
        String label = cellData.getStringValue();
        String value;

        if (StringUtils.isBlank(label)) {
            return null;
        }

        // 处理多选值
        if (label.contains(anno.separator())) {
            String[] labels = label.split(anno.separator());
            List<String> values = new ArrayList<>();
            for (String l : labels) {
                String v;
                if (StringUtils.isBlank(type)) {
                    v = ExcelUtil.reverseByExp(l.trim(), anno.readConverterExp(), anno.separator());
                } else {
                    v = SpringUtils.getBean(DictService.class).getDictValue(type, l.trim(), anno.separator());
                }
                if (StringUtils.isNotEmpty(v)) {
                    values.add(v);
                }
            }
            value = String.join(anno.separator(), values);
        } else {
            // 单值处理
            if (StringUtils.isBlank(type)) {
                value = ExcelUtil.reverseByExp(label, anno.readConverterExp(), anno.separator());
            } else {
                value = SpringUtils.getBean(DictService.class).getDictValue(type, label, anno.separator());
            }
        }

        return Convert.convert(contentProperty.getField().getType(), value);
    }

    @Override
    public WriteCellData<String> convertToExcelData(Object object, ExcelContentProperty contentProperty, GlobalConfiguration globalConfiguration) {
        if (ObjectUtil.isNull(object)) {
            return new WriteCellData<>("");
        }

        ExcelDictFormat anno = getAnnotation(contentProperty.getField());
        String type = anno.dictType();
        String value = Convert.toStr(object);
        String label;

        if (StringUtils.isBlank(value)) {
            return new WriteCellData<>("");
        }

        // 处理多选值
        if (value.contains(anno.separator())) {
            String[] values = value.split(anno.separator());
            List<String> labels = new ArrayList<>();
            for (String v : values) {
                String l;
                if (StringUtils.isBlank(type)) {
                    l = ExcelUtil.convertByExp(v.trim(), anno.readConverterExp(), anno.separator());
                } else {
                    l = SpringUtils.getBean(DictService.class).getDictLabel(type, v.trim(), anno.separator());
                }
                if (StringUtils.isNotEmpty(l)) {
                    labels.add(l);
                }
            }
            label = String.join(anno.separator(), labels);
        } else {
            // 单值处理
            if (StringUtils.isBlank(type)) {
                label = ExcelUtil.convertByExp(value, anno.readConverterExp(), anno.separator());
            } else {
                label = SpringUtils.getBean(DictService.class).getDictLabel(type, value, anno.separator());
            }
        }

        return new WriteCellData<>(label);
    }

    private ExcelDictFormat getAnnotation(Field field) {
        return AnnotationUtil.getAnnotation(field, ExcelDictFormat.class);
    }
}
