package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.ZjkExpertMoneyInfo;
import org.dromara.zjk.domain.bo.ZjkExpertMoneyInfoBo;
import org.dromara.zjk.domain.vo.ZjkExpertMoneyInfoVO;
import org.dromara.zjk.service.ZjkExpertMoneyInfoService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.dromara.zjk.zwy.annotation.EncryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import  org.dromara.zjk.domain.vo.ZjkExpertVo;
import javax.annotation.Resource;
import java.util.List;

/**
 * 专家费用
 *
 * @author : yanguohua
 * @version : 1.0
 * @date : 2024/10/30 16:57
 */
@RestController
@RequestMapping("/moneyInfo")
public class ZjkExpertMoneyInfoController extends BaseController {

    @Resource
    private ZjkExpertMoneyInfoService zjkExpertMoneyInfoService;

    /**
     * 专家费用列表
     */
    @DecryptOperation
    @GetMapping("/list")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
    }, mode = SaMode.OR)
    public TableDataInfo<ZjkExpertMoneyInfo> page(ZjkExpertMoneyInfoBo bo, PageQuery query) {
        return zjkExpertMoneyInfoService.queryPageList(bo, query);
    }

    /**
     * 结算
     *
     * @param bo
     * @return
     */
    @PostMapping("/update-settlement")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
    }, mode = SaMode.OR)
    public R updateSettlement(@RequestBody ZjkExpertMoneyInfoBo bo) {
        return toAjax(zjkExpertMoneyInfoService.updateMoneyInfo(bo));
    }

    /**
     * 支付
     *
     * @param bo
     * @return
     */
    @PostMapping("/update-pay")
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
    }, mode = SaMode.OR)
    public R update(@RequestBody ZjkExpertMoneyInfoBo bo) {
        return toAjax(zjkExpertMoneyInfoService.updateMoneyInfoPay(bo));
    }

    /**
     * 导出专家结算费用管理列表
     */
    @DecryptOperation
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    }, mode = SaMode.OR)
    @PostMapping("/export")
    public void export(ZjkExpertMoneyInfoBo bo, HttpServletResponse response) {
        List<ZjkExpertMoneyInfoVO> list = zjkExpertMoneyInfoService.queryList(bo);
        ExcelUtil.exportExcel(list, "导出", ZjkExpertMoneyInfoVO.class, response);
    }

    /**
     * 获取专家结算费用管理详细信息
     *
     * @param moneyId 主键
     */
    @DecryptOperation
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    }, mode = SaMode.OR)
    @GetMapping("/{moneyId}")
    public R<ZjkExpertMoneyInfo> getInfo(@NotNull(message = "主键不能为空")
                                         @PathVariable Long moneyId) {
        return R.ok(zjkExpertMoneyInfoService.queryById(moneyId));
    }

    /**
     * 新增专家结算费用管理
     */
    @EncryptOperation
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    }, mode = SaMode.OR)
    @RepeatSubmit()
    @PostMapping("/add")
    public R<Void> add(@RequestBody ZjkExpertMoneyInfoBo bo) {
        return toAjax(zjkExpertMoneyInfoService.insertByBo(bo));
    }

    /**
     * 修改专家结算费用管理
     */
    @EncryptOperation
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    }, mode = SaMode.OR)
    @RepeatSubmit()
    @PostMapping("/edit")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkExpertMoneyInfoBo bo) {
        return toAjax(zjkExpertMoneyInfoService.updateByBo(bo));
    }

    /**
     * 删除专家结算费用管理
     *
     * @param moneyIds 主键串
     */
    @SaCheckRole(value = {
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    }, mode = SaMode.OR)
    @GetMapping("/delete/{moneyIds}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] moneyIds) {
        return toAjax(zjkExpertMoneyInfoService.deleteWithValidByIds(List.of(moneyIds), true));
    }

    /**
     * 通过项目id和阶段获取结算信息列表
     *
     * @param productId
     * @param reviewSn
     * @return
     */
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    @GetMapping("/settleList")
    public R<List<ZjkExpertMoneyInfo>> getJsInfo(@NotNull(message = "项目不能为空")
                                                 Long productId, @NotNull(message = "阶段不能为空")
                                                 Integer reviewSn, @NotNull(message = "专家不能为空")
                                                 String expertId) {
        return R.ok(zjkExpertMoneyInfoService.getJsList(productId, reviewSn, expertId));
    }
    /**
     * 通过项目id和阶段获取参评专家列表
     * @param productId
     * @param reviewSn
     * @param toSettle 1:新增结算单带出的专家
     * @return
     */
    @DecryptOperation
    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY
    }, mode = SaMode.OR)
    @GetMapping("/expertList")
    public R<List<ZjkExpertVo>> getExpertList(@NotNull(message = "项目不能为空")
                                                 Long productId, @NotNull(message = "阶段不能为空")
                                                 Integer reviewSn,String toSettle) {
        if(toSettle!=null){
            return R.ok(zjkExpertMoneyInfoService.getExpertListForSettle(productId, reviewSn));

        }else{
            //发票管理的人员带出
            return R.ok(zjkExpertMoneyInfoService.getExpertList(productId, reviewSn));

        }
    }
}
