package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkProjectEvaluationBo;
import org.dromara.zjk.domain.vo.ZjkProjectEvaluationVo;
import org.dromara.zjk.service.IZjkProjectEvaluationService;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 项目总结评价
 * 前端访问路由地址为:/jyk/projectEvaluation
 *
 * @author ygh
 * @date 2024-11-08
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/projectEvaluation")
public class ZjkProjectEvaluationController extends BaseController {

    private final IZjkProjectEvaluationService zjkProjectEvaluationService;

    /**
     * 查询项目总结评价列表
     */
    @SaCheckPermission("jyk:projectEvaluation:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkProjectEvaluationVo> list(ZjkProjectEvaluationBo bo, PageQuery pageQuery) {
        return zjkProjectEvaluationService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出项目总结评价列表
     */
    @SaCheckPermission("jyk:projectEvaluation:export")
    @Log(title = "项目总结评价", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkProjectEvaluationBo bo, HttpServletResponse response) {
        List<ZjkProjectEvaluationVo> list = zjkProjectEvaluationService.queryList(bo);
        ExcelUtil.exportExcel(list, "项目总结评价", ZjkProjectEvaluationVo.class, response);
    }

    /**
     * 获取项目总结评价详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("jyk:projectEvaluation:query")
    @GetMapping("/{id}")
    public R<ZjkProjectEvaluationVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkProjectEvaluationService.queryById(id));
    }

    /**
     * 新增项目总结评价
     */
    @SaCheckRole(value =
        {
            TenantConstants.XM_ROLE_KEY,
            TenantConstants.TENANT_ADMIN_ROLE_KEY,
        }
        , mode = SaMode.OR)
    @Log(title = "项目总结评价", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping("/insert")
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkProjectEvaluationBo bo) {
        return toAjax(zjkProjectEvaluationService.insertByBo(bo));
    }

    /**
     * 修改项目总结评价
     */
    @SaCheckPermission("jyk:projectEvaluation:edit")
    @Log(title = "项目总结评价", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkProjectEvaluationBo bo) {
        return toAjax(zjkProjectEvaluationService.updateByBo(bo));
    }

    /**
     * 删除项目总结评价
     *
     * @param ids 主键串
     */
    @SaCheckPermission("jyk:projectEvaluation:remove")
    @Log(title = "项目总结评价", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkProjectEvaluationService.deleteWithValidByIds(List.of(ids), true));
    }
}
