package org.dromara.zjk.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import cn.hutool.core.bean.BeanUtil;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.apache.commons.math3.stat.descriptive.summary.Product;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.domain.bo.ZjkProductBo;
import org.dromara.zjk.domain.bo.ZjkReviewPhaseBo;
import org.dromara.zjk.domain.bo.ZjkReviewProductBo;
import org.dromara.zjk.domain.vo.ZjkReviewExportVo;
import org.dromara.zjk.domain.vo.ZjkReviewPhaseVo;
import org.dromara.zjk.domain.vo.ZjkReviewProductVo;
import org.dromara.zjk.service.IZjkReviewPhaseService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.dromara.zjk.zwy.annotation.EncryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 抽取条件
 * 前端访问路由地址为:/jyk/reviewPhase
 *
 * @author ygh
 * @date 2024-11-01
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/reviewPhase")
public class ZjkReviewPhaseController extends BaseController {

    private final IZjkReviewPhaseService zjkReviewPhaseService;

    /**
     * 查询抽取条件列表
     */
    @DecryptOperation
    @SaCheckPermission("jyk:reviewPhase:list")
    @GetMapping("/list")
    public TableDataInfo<ZjkReviewPhaseVo> list(ZjkReviewPhaseBo bo, PageQuery pageQuery) {
        return zjkReviewPhaseService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出抽取条件列表
     */
    @DecryptOperation
    @SaCheckPermission("jyk:reviewPhase:export")
    @Log(title = "抽取条件", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(ZjkReviewPhaseBo bo, HttpServletResponse response) {
        List<ZjkReviewPhaseVo> list = zjkReviewPhaseService.queryList(bo);
        ExcelUtil.exportExcel(list, "抽取条件", ZjkReviewPhaseVo.class, response);
    }

    /**
     * 获取抽取条件详细信息
     *
     * @param tenantId 主键
     */
    @DecryptOperation
    @SaCheckPermission("jyk:reviewPhase:query")
    @GetMapping("/{tenantId}")
    public R<ZjkReviewPhaseVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable String tenantId) {
        return R.ok(zjkReviewPhaseService.queryById(tenantId));
    }

    /**
     * 新增抽取条件
     */
    @EncryptOperation
    @SaCheckPermission("jyk:reviewPhase:add")
    @Log(title = "抽取条件", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkReviewPhaseBo bo) {
        return toAjax(zjkReviewPhaseService.insertByBo(bo));
    }

    /**
     * 修改抽取条件
     */
    @EncryptOperation
    @SaCheckPermission("jyk:reviewPhase:edit")
    @Log(title = "抽取条件", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkReviewPhaseBo bo) {
        return toAjax(zjkReviewPhaseService.updateByBo(bo));
    }

    /**
     * 删除抽取条件
     *
     * @param tenantIds 主键串
     */
    @SaCheckPermission("jyk:reviewPhase:remove")
    @Log(title = "抽取条件", businessType = BusinessType.DELETE)
    @DeleteMapping("/{tenantIds}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable String[] tenantIds) {
        return toAjax(zjkReviewPhaseService.deleteWithValidByIds(List.of(tenantIds), true));
    }
    @DecryptOperation
    @SaCheckPermission("zjk:reviewproduct:list")
    @GetMapping("/review/product/list")
    public TableDataInfo<ZjkReviewProductVo> reviewProductList(ZjkReviewProductBo bo, PageQuery pageQuery) {
        return zjkReviewPhaseService.reviewProductList(bo, pageQuery);
    }

    /**
     * 发票/结算 获取项目列表
     * @return
     */
    @GetMapping("/invoice/productList")
    public R<List<ZjkProductBo>> invoiceProductList() {
        return zjkReviewPhaseService.invoiceProductList();
    }

    @DecryptOperation
    @SaCheckPermission("zjk:reviewproduct:export")
    @PostMapping("/review/export")
    public void  reviewExport(ZjkReviewProductBo bo, HttpServletResponse response) {
        List<ZjkReviewProductVo> zjkReviewProductVos = zjkReviewPhaseService.reviewExport(bo);
        //List<ZjkReviewExportVo> zjkReviewExportVo = BeanUtil.copyToList(zjkReviewProductVos, ZjkReviewExportVo.class);
        ExcelUtil.exportExcel(zjkReviewProductVos, "评审项目", ZjkReviewProductVo.class, response);
    }


    @DecryptOperation
    @GetMapping("/getInfo/{productId}")
    public R<ZjkReviewProductVo> getInfo(@NotNull(message = "项目id不能为空")
                                   @PathVariable Long productId) {
        ZjkReviewProductVo zjkReviewProductVos = zjkReviewPhaseService.reviewListById(productId);
        return R.ok(zjkReviewProductVos);
    }

    /**
     * 根据专家 id 获取合作项目信息
     * @param bo
     * @param pageQuery
     * @return
     */
    @DecryptOperation
    @SaCheckPermission("zjk:reviewproduct:list")
    @GetMapping("/getReviewProductListByExpert")
    public TableDataInfo<ZjkReviewProductVo> getReviewProductListByExpert(ZjkReviewProductBo bo, PageQuery pageQuery) {
        return zjkReviewPhaseService.getReviewProductListByExpert(bo, pageQuery);
    }

    @SaCheckRole(value = {
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.XM_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/getReview/count")
    public R getReviewCount() {
        return R.ok("success",zjkReviewPhaseService.getReviewCount());
    }

}
