package org.dromara.zjk.expert.expertPatent.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.utils.MapstructUtils;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.zjk.expert.expertPatent.domain.ZjkExpertPatent;
import org.dromara.zjk.expert.expertPatent.domain.bo.ZjkExpertPatentBo;
import org.dromara.zjk.expert.expertPatent.domain.vo.ZjkExpertPatentVo;
import org.dromara.zjk.expert.expertPatent.mapper.ZjkExpertPatentMapper;
import org.dromara.zjk.expert.expertPatent.service.IZjkExpertPatentService;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 专家专利信息Service业务层处理
 *
 * @author zhangzhou
 * @date 2024-11-10
 */
@RequiredArgsConstructor
@Service
public class ZjkExpertPatentServiceImpl implements IZjkExpertPatentService {

    private final ZjkExpertPatentMapper baseMapper;

    /**
     * 查询专家专利信息
     *
     * @param id 主键
     * @return 专家专利信息
     */
    @Override
    public ZjkExpertPatentVo queryById(Long id){
        return baseMapper.selectVoById(id);
    }

    /**
     * 分页查询专家专利信息列表
     *
     * @param bo        查询条件
     * @param pageQuery 分页参数
     * @return 专家专利信息分页列表
     */
    @Override
    public TableDataInfo<ZjkExpertPatentVo> queryPageList(ZjkExpertPatentBo bo, PageQuery pageQuery) {
        Page<ZjkExpertPatentVo> result = baseMapper.selectVoListJoinExpertId(pageQuery.build(), bo);
        return TableDataInfo.build(result);
    }

    /**
     * 查询符合条件的专家专利信息列表
     *
     * @param bo 查询条件
     * @return 专家专利信息列表
     */
    @Override
    public List<ZjkExpertPatentVo> queryList(ZjkExpertPatentBo bo) {
        LambdaQueryWrapper<ZjkExpertPatent> lqw = buildQueryWrapper(bo);
        return baseMapper.selectVoList(lqw);
    }

    private LambdaQueryWrapper<ZjkExpertPatent> buildQueryWrapper(ZjkExpertPatentBo bo) {
        Map<String, Object> params = bo.getParams();
        LambdaQueryWrapper<ZjkExpertPatent> lqw = Wrappers.lambdaQuery();
        lqw.eq(bo.getExpertId() != null, ZjkExpertPatent::getExpertId, bo.getExpertId());
        lqw.eq(StringUtils.isNotBlank(bo.getPatentNumber()), ZjkExpertPatent::getPatentNumber, bo.getPatentNumber());
        lqw.eq(StringUtils.isNotBlank(bo.getNameInvention()), ZjkExpertPatent::getNameInvention, bo.getNameInvention());
        lqw.eq(StringUtils.isNotBlank(bo.getPatentPath()), ZjkExpertPatent::getPatentPath, bo.getPatentPath());
        return lqw;
    }

    /**
     * 新增专家专利信息
     *
     * @param bo 专家专利信息
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByBo(ZjkExpertPatentBo bo) {
        ZjkExpertPatent add = MapstructUtils.convert(bo, ZjkExpertPatent.class);
        validEntityBeforeSave(add);
        boolean flag = baseMapper.insert(add) > 0;
        if (flag) {
            bo.setId(add.getId());
        }
        return flag;
    }

    /**
     * 修改专家专利信息
     *
     * @param bo 专家专利信息
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByBo(ZjkExpertPatentBo bo) {
        ZjkExpertPatent update = MapstructUtils.convert(bo, ZjkExpertPatent.class);
        validEntityBeforeSave(update);
        return baseMapper.updateById(update) > 0;
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(ZjkExpertPatent entity){
        //TODO 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除专家专利信息信息
     *
     * @param ids     待删除的主键集合
     * @param isValid 是否进行有效性校验
     * @return 是否删除成功
     */
    @Override
    public Boolean deleteWithValidByIds(Collection<Long> ids, Boolean isValid) {
        if(isValid){
            //TODO 做一些业务上的校验,判断是否需要校验
        }
        return baseMapper.deleteByIds(ids) > 0;
    }

    @Override
    public void handleExpertPatents(List<ZjkExpertPatent> zjkExpertPatentList, Long expertId) {
        baseMapper.delete(new LambdaQueryWrapper<ZjkExpertPatent>()
            .eq(ZjkExpertPatent::getExpertId, expertId)
        );
        if (zjkExpertPatentList != null && !zjkExpertPatentList.isEmpty()) {
            zjkExpertPatentList.forEach(zjkExpertAwards -> {
                zjkExpertAwards.setExpertId(expertId);
            });
            baseMapper.insertOrUpdateBatch(zjkExpertPatentList);
        }



    }

    @Override
    public List<ZjkExpertPatent> populateExpertPatentsList(Long expertId) {
        List<ZjkExpertPatent> zjkExpertPatentList = baseMapper.selectList(
            new LambdaQueryWrapper<ZjkExpertPatent>()
                .eq(ZjkExpertPatent::getExpertId, expertId)
        );
        return zjkExpertPatentList.isEmpty() ? new ArrayList<>() : zjkExpertPatentList;
    }

    @Override
    public void deleteByProductId(Long productReviewId) {
        LambdaQueryWrapper<ZjkExpertPatent> wrapper = Wrappers.lambdaQuery();
        wrapper.eq(ZjkExpertPatent::getProductId, productReviewId);
        baseMapper.delete(wrapper);
    }

    @Override
    public boolean saveBatch(List<ZjkExpertPatent> zjkExpertPatentsList) {
        return baseMapper.insertBatch(zjkExpertPatentsList);
    }

    @Override
    public List<ZjkExpertPatent> selectListByProductIds(Long zjkProductReviewId) {
        LambdaQueryWrapper<ZjkExpertPatent> wrapper = Wrappers.lambdaQuery();
        wrapper.in(ZjkExpertPatent::getProductId, zjkProductReviewId);
        return  baseMapper.selectList(wrapper);
    }

    @Override
    public List<ZjkExpertPatentVo> patentList(Long productId, String reviewSn) {
        return baseMapper.patentList(productId,reviewSn);
    }

    @Override
    public Long getExpertPatentCount() {
        return baseMapper.getExpertPatentCount();
    }

    @Override
    public List<ZjkExpertPatent> selectByExpertId(Long expertId) {
        return baseMapper.selectByExpertId(expertId);
    }

}
