package org.dromara.zjk.handler;

import lombok.extern.slf4j.Slf4j;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.utils.StringUtils;
import org.dromara.zjk.utils.ApiThreadLocal;
import org.dromara.zjk.utils.EncryptionUtil;
import org.springframework.core.MethodParameter;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;

/**
 * @author : yanguohua
 * @version : 1.0
 * @date : 2025/8/27 10:11
 */
@RestControllerAdvice(basePackages = "org.dromara.zjk.controller")
@Slf4j
public class ApiEncryptResponseAdvice implements ResponseBodyAdvice<R<String>> {

    @Override
    public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
        // 只处理返回类型为 R<String> 的接口
        return R.class.isAssignableFrom(returnType.getParameterType());
    }

    @Override
    public R<String> beforeBodyWrite(R<String> body, MethodParameter returnType, MediaType selectedContentType,
                                     Class<? extends HttpMessageConverter<?>> selectedConverterType,
                                     ServerHttpRequest request, ServerHttpResponse response) {
        // 判断是否需要加密（可以根据注解、路径等条件判断）
        if (body != null && body.getCode() == 200 ) {
            String ak = ApiThreadLocal.getAk();
            if (StringUtils.isNotEmpty(ak)) {
                try {
                    // 对数据进行加密
                    String encryptedData = EncryptionUtil.encrypt(body.getData(), ak);
                    // 替换原始数据为加密数据
                    return R.ok(body.getMsg(), encryptedData);
                } catch (Exception e) {
                    // 加密失败记录日志
                    log.error("数据加密失败", e);
                }
            }
        }
        return body;
    }
}
