package org.dromara.zjk.learning_world.controller;

import cn.dev33.satoken.annotation.SaCheckRole;
import cn.dev33.satoken.annotation.SaMode;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import lombok.RequiredArgsConstructor;
import org.dromara.common.core.constant.TenantConstants;
import org.dromara.common.core.domain.R;
import org.dromara.common.core.validate.AddGroup;
import org.dromara.common.core.validate.EditGroup;
import org.dromara.common.excel.utils.ExcelUtil;
import org.dromara.common.idempotent.annotation.RepeatSubmit;
import org.dromara.common.log.annotation.Log;
import org.dromara.common.log.enums.BusinessType;
import org.dromara.common.mybatis.core.page.PageQuery;
import org.dromara.common.mybatis.core.page.TableDataInfo;
import org.dromara.common.web.core.BaseController;
import org.dromara.zjk.learning_world.domain.bo.ZjkLearningCommontBo;
import org.dromara.zjk.learning_world.domain.vo.ZjkLearningCommontVo;
import org.dromara.zjk.learning_world.service.IZjkLearningCommontService;
import org.dromara.zjk.zwy.annotation.DecryptOperation;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 评论
 * 前端访问路由地址为:/zjk/learningCommont
 *
 * @author ygh
 * @date 2024-11-11
 */
@Validated
@RequiredArgsConstructor
@RestController
@RequestMapping("/learningCommont")
public class ZjkLearningCommontController extends BaseController {

    private final IZjkLearningCommontService zjkLearningCommontService;

    /**
     * 查询评论列表
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/list")
    public TableDataInfo<ZjkLearningCommontVo> list(ZjkLearningCommontBo bo, PageQuery pageQuery) {
        return zjkLearningCommontService.queryPageList(bo, pageQuery);
    }

    /**
     * 导出评论列表
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @DecryptOperation
    @PostMapping("/export")
    public void export(ZjkLearningCommontBo bo, HttpServletResponse response) {
        List<ZjkLearningCommontVo> list = zjkLearningCommontService.queryList(bo);
        ExcelUtil.exportExcel(list, "评论", ZjkLearningCommontVo.class, response);
    }

    /**
     * 获取评论详细信息
     *
     * @param id 主键
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @GetMapping("/{id}")
    public R<ZjkLearningCommontVo> getInfo(@NotNull(message = "主键不能为空")
                                     @PathVariable Long id) {
        return R.ok(zjkLearningCommontService.queryById(id));
    }

    /**
     * 新增评论
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "评论", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public R<Void> add(@Validated(AddGroup.class) @RequestBody ZjkLearningCommontBo bo) {
        return toAjax(zjkLearningCommontService.insertByBo(bo));
    }

    /**
     * 修改评论
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "评论", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update")
    public R<Void> edit(@Validated(EditGroup.class) @RequestBody ZjkLearningCommontBo bo) {
        return toAjax(zjkLearningCommontService.updateByBo(bo));
    }


    /**
     * 修改评论
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "评论", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PostMapping("/update/status")
    public R<Void> editStatus(@Validated(EditGroup.class) @RequestBody ZjkLearningCommontBo bo) {
        return toAjax(zjkLearningCommontService.updateStatuso(bo));
    }

    /**
     * 删除评论
     *
     * @param ids 主键串
     */
    @SaCheckRole(value={
        TenantConstants.JG_ROLE_KEY,
        TenantConstants.SUPER_ADMIN_ROLE_KEY,
        TenantConstants.TENANT_ADMIN_ROLE_KEY,
        TenantConstants.EXPERT_ROLE_KEY,
    },mode = SaMode.OR)
    @Log(title = "评论", businessType = BusinessType.DELETE)
    @GetMapping("/deleted/{ids}")
    public R<Void> remove(@NotEmpty(message = "主键不能为空")
                          @PathVariable Long[] ids) {
        return toAjax(zjkLearningCommontService.deleteWithValidByIds(List.of(ids), true));
    }
}
